﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using Curse.Friends.Enums;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.NotificationContracts;
using System.Diagnostics;

namespace Curse.Friends.Data
{

    [TableDefinition(TableName = "User", KeySpace = "CurseVoice", ReplicationMode = ReplicationMode.None)]
    [DebuggerDisplay("{UserID} - {Username}")]
    public class User : BaseTable<User>, IUserIdentity
    {
        public const int CurrentGameStatusMessageMaxLength = 256;
        public const int AvatarUrlMaxLength = 256;
        public const int NameMaxLength = 64;
        public const int CityMaxLength = 64;
        public const int StateMaxLength = 64;
        public const int CountryCodeMaxLength = 2;
        public const int AboutMeMaxLength = 200;
        public const string AvatarUrlPath = "users";

        public const int ManualAccountSyncThrottleSeconds = 60;

        public static readonly TimeSpan AccountRenameThrottleTimespan = TimeSpan.FromDays(60);
        public static readonly TimeSpan SocialSyncInterval = TimeSpan.FromDays(7);
        public static readonly TimeSpan AccountSyncInterval = TimeSpan.FromMinutes(30);

        [Column("UserID", KeyOrdinal = 1)]
        public int UserID
        {
            get;
            set;
        }

        [Column("Username")]
        public string Username
        {
            get;
            set;
        }

        [Column("DisplayName")]
        public string DisplayName
        {
            get;
            set;
        }

        /// <summary>
        /// User's don't have nicknames. Always null!
        /// </summary>
        string IUserIdentity.Nickname
        {
            get { return null; }
        }

        public string ResolvedName
        {
            get { return this.GetResolvedName(); }
        }
       
        [Column("ConnStatus")]        
        public UserConnectionStatus ConnectionStatus
        {
            get;
            set;
        }

        [Column("LastStatus")]
        public UserConnectionStatus LastConnectionStatus
        {
            get;
            set;
        }
        
        [Column("CustStatMsg")]
        public string CustomStatusMessage
        {
            get;
            set;
        }

        [Column("CustStatTime")]
        public DateTime CustomStatusTimestamp
        {
            get;
            set;
        }

        [Column("FriendCount")]
        public int FriendCount
        {
            get;
            set;

        }

        [Column("LastGameID")]
        public int LastGameID
        {
            get;
            set;
        }

        [Column("CurGameState")]
        public int CurrentGameState
        {
            get;
            set;
        }

        [Column("CurGameStatus")]
        public string CurrentGameStatusMessage
        {
            get;
            set;
        }

        [Column("CurGameTime")]
        public DateTime CurrentGameTimestamp
        {
            get;
            set;
        }

        [Column("LastFriendsSug")]
        public DateTime LastFriendsSuggestion
        {
            get;
            set;
        }

        [Column("HasSyncedAcct")]
        public bool HasSyncedAccount
        {
            get;
            set;
        }

        [Column("LastStreamSync")]
        public DateTime LastAccountSync
        {
            get;
            set;
        }

        [Column("LastManualSync")]
        public DateTime LastManualAccountSync
        {
            get; 
            set;
        }

        [Column("LastSocialSync")]
        public DateTime LastSocialSync
        {
            get;
            set;
        }

        #region User Profile Fields

        [Column("Name")]
        public string Name
        {
            get;
            set;
        }

        [Column("AvatarUrl")]
        public string AvatarUrl
        {
            get;
            set;
        }

        [Column("City")]
        public string City
        {
            get;
            set;
        }

        [Column("State")]
        public string State
        {
            get;
            set;
        }

        [Column("CountryCode")]
        public string CountryCode
        {
            get;
            set;
        }

        [Column("AboutMe")]
        public string AboutMe
        {
            get;
            set;
        }

        #endregion

        [Column("IndexMode")]
        public IndexMode IndexMode
        {
            get;
            set;
        }       

        /// <summary>
        /// A user configurable preference which determines under what conditions a user receives
        /// push notifications for group messages.
        /// </summary>
        [Column("GroupPush")]
        public PushNotificationPreference GroupMessagePushPreference { get; set; }

        /// <summary>
        /// A user configurable preference which determines under what conditions a user receives
        /// push notifications for friend messages.
        /// </summary>
        [Column("FriendPush")]
        public PushNotificationPreference FriendMessagePushPreference { get; set; }

        [Column("InvitesPush")]
        public bool FriendRequestPushEnabled { get; set; }

        [Column("MentionsPush")]
        public bool? MentionsPushEnabled { get; set; }

        [Column("SyncSearch")]
        public DateTime LastFriendSyncSearch { get; set; }

        [Column("SyncUpload")]
        public DateTime LastFriendSyncUpload { get; set; }


        [Column("IsTempAccount")]
        public bool IsTempAccount
        {
            get;
            set;
        }

        [Column("LastNameChange")]
        public DateTime LastNameChangeDate { get; set; }

        [Column("TwitchID")]
        public string TwitchID { get; set; }
        
        [Column("AvatarDate")]
        public DateTime AvatarDate { get; set; }

        [Column("IsMerged")]
        public bool IsMerged { get; set; }

        [Column("EmailAddress")]
        public string EmailAddress { get; set; }

        [Column("EmailVerified")]
        public bool EmailVerified { get; set; }

        [Column("CurGameID")]
        public int CurrentGameID { get; set; }

        [Column("WatchChanID")]
        public string WatchingChannelID { get; set; }


        [Column("IsCasting")]
        public bool IsBroadcasting { get; set; }

        [Column("CastGameID")]
        public int BroadcastingGameID { get; set; }

        [Column("ActivityTime")]
        public object ActivityTimestamp { get; set; }

        [Column("Capabilities")]
        public ClientCapability Capabilities { get; set; }

        [Column("IsProvisional")]
        public bool IsProvisional { get; set; }

        #region Local Status
        [Column("ConnStatusLoc")]
        public UserConnectionStatus ConnectionStatusLocal { get; set; }

        [Column("WatchIDLoc")]
        public string WatchingChannelIDLocal { get; set; }

        [Column("GameIDLoc")]
        public int CurrentGameIDLocal { get; set; }

        [Column("GameStateLoc")]
        public int CurrentGameStateLocal { get; set; }

        [Column("GameMsgLoc")]
        public string CurrentGameStatusMessageLocal { get; set; }

        [Column("GameTimeLoc")]
        public DateTime CurrentGameTimestampLocal { get; set; }

        [Column("IsBroadcastLoc")]
        public bool IsBroadcastingLocal { get; set; }

        [Column("BroadcastIDLoc")]
        public int BroadcastingGameIDLocal { get; set; }

        #endregion


        public string TwitchOrCurseID
        {
            get { return IsMerged ? TwitchID : UserID.ToString(); }
        }


        public void UpdateFriendCount(int regionID, Friendship[] friends)
        {
            FriendCount = friends.Length;
            Update(p => p.FriendCount);
        }

        public void UpdateUserStats(Friendship[] friends)
        {
            var friendIDs = friends.Select(p => p.OtherUserID).ToArray();
            var stats = GetStatistics();
            stats.FriendIDs = new HashSet<int>(friendIDs);
            stats.Update(s => s.FriendIDs);
        }

        protected override void Validate()
        {
            if (string.IsNullOrEmpty(Username))
            {
                throw new Exception("Username cannot be null or empty");
            }

            if (UserID < 1)
            {
                throw new Exception("UserID must be 1 or higher");
            }
        }

        public UserConnectionStatus GetDisplayStatus()
        {
            return GetDisplayStatus(ConnectionStatus);
        }

        public static UserConnectionStatus GetDisplayStatus(UserConnectionStatus status)
        {
            UserConnectionStatus effectiveStatus = status;
            switch (status)
            {
                case UserConnectionStatus.Invisible:
                    effectiveStatus = UserConnectionStatus.Offline;
                    break;

                case UserConnectionStatus.Idle:
                    effectiveStatus = UserConnectionStatus.Away;
                    break;

            }

            return effectiveStatus;
        }

        public static MutualFriend[] GetMutualFriendIDs(int myUserID, int otherUserID, HashSet<int> excludeIDs = null)
        {
            var myFriendIDs = UserStatistics.GetByUserOrDefault(myUserID).FriendIDs;
            var otherFriendIDs = UserStatistics.GetByUserOrDefault(otherUserID).FriendIDs;

            var mutualFriends = myFriendIDs.Where(p => otherFriendIDs.Contains(p)).ToArray();
            var list = new List<MutualFriend>();

            var mutualFriendStatistics = UserStatistics.MultiGetLocal(mutualFriends.Select(p => new KeyInfo(p)));

            foreach (var friendStatistics in mutualFriendStatistics)
            {
                var mutualFriend = new MutualFriend(friendStatistics.UserID);

                foreach (var f in friendStatistics.FriendIDs.Where(p => myFriendIDs.Contains(p)))
                {
                    mutualFriend.AddMutualUser(f);
                }

                list.Add(mutualFriend);
            }

            return list.ToArray();
        }

        public static Guid[] GetMutualGroupIDs(int myUserID, int otherUserID)
        {
            // Get the user's group memberships
            var userGroupIDs = new HashSet<Guid>(GroupMember.GetAllByUserID(myUserID).Where(p => p.IsRootGroup).Select(p => p.GroupID));

            // Get the other user's group memberships
            var otherUserGroupIDs = new HashSet<Guid>(GroupMember.GetAllByUserID(otherUserID).Where(p => p.IsRootGroup).Select(p => p.GroupID));

            // Return only those that are in common
            return otherUserGroupIDs.Where(id => userGroupIDs.Contains(id)).ToArray();
        }

        public static int[] GetMutualFriends(int myUserID, int otherUserID, HashSet<int> excludeIDs = null)
        {
            var myFriendIDs = UserStatistics.GetByUserOrDefault(myUserID).FriendIDs;
            var otherFriendIDs = UserStatistics.GetByUserOrDefault(otherUserID).FriendIDs;

            var mutualFriends = myFriendIDs.Where(p => otherFriendIDs.Contains(p)).ToArray();
            if (excludeIDs != null && excludeIDs.Any())
            {
                return mutualFriends.Where(p => !excludeIDs.Contains(p)).ToArray();
            }
            return mutualFriends;
        }

        public UserStatistics GetStatistics()
        {
            var stats = UserStatistics.GetLocal(UserID);

            if (stats == null)
            {
                stats = new UserStatistics { UserID = UserID, FriendIDs = new HashSet<int>() };
                stats.InsertLocal();
            }

            return stats;

        }

        public void UpdateAvatarTimestamp(DateTime avatarDate)
        {
            var notify = false;
            if (AvatarDate != avatarDate)
            {
                AvatarDate = avatarDate;
                Update(u => u.AvatarDate);
                notify = true;
            }

            var timestamp = avatarDate.ToEpochMilliseconds();
            var stats = GetStatistics();
            if (stats.AvatarTimestamp != timestamp)
            {
                stats.AvatarTimestamp = timestamp;
                stats.Update(s => s.AvatarTimestamp);
            }

            if (notify)
            {
                RegionalUserChangeResolver.Create(UserID);
            }

        }

        public void UpdateStatistics(long tokenTimestamp = 0)
        {
            var privacy = GetPrivacy();
            var stats = GetStatistics();
            var changes = new List<Expression<Func<UserStatistics, object>>>();

            if (stats.Username != Username)
            {
                stats.Username = Username;
                changes.Add(p => p.Username);
            }

            if (stats.DisplayName != DisplayName)
            {
                stats.DisplayName = DisplayName;
                changes.Add(p => p.DisplayName);
            }

            // Connnection Status
            if (stats.ConnectionStatus != ConnectionStatus)
            {
                // Coalesce Idle/Invisible to the appropriate display status
                stats.ConnectionStatus = GetDisplayStatus(ConnectionStatus);
                stats.ConnectStatusTimestamp = CustomStatusTimestamp.SafeToEpochMilliseconds();
                if (ConnectionStatus == UserConnectionStatus.Offline)
                {
                    stats.DateLastSeen = DateTime.UtcNow.ToEpochMilliseconds();
                    changes.Add(p => p.DateLastSeen);
                }
                changes.Add(p => p.ConnectionStatus);
                changes.Add(p => p.ConnectStatusTimestamp);
            }

            if (stats.StatusMessage != CustomStatusMessage)
            {
                stats.StatusMessage = CustomStatusMessage;
                changes.Add(p => p.StatusMessage);
            }

            // If user's status is invisible, clear out rich presence
            if (ConnectionStatus == UserConnectionStatus.Invisible || privacy.ShareActivityPrivacy == ShareActivityPrivacy.Hide) // Clear stats
            {
                if (stats.CurrentGameID != 0)
                {
                    stats.CurrentGameID = 0;                    
                    changes.Add(p => p.CurrentGameID);
                }

                if (stats.WatchingChannelID != string.Empty)
                {
                    stats.WatchingChannelID = string.Empty;
                    changes.Add(p => p.WatchingChannelID);
                }

                if (stats.BroadcastingGameID != 0)
                {
                    stats.BroadcastingGameID = 0;
                    changes.Add(p => p.BroadcastingGameID);
                }

                if (stats.IsBroadcasting)
                {
                    stats.IsBroadcasting = false;
                    changes.Add(p => p.IsBroadcasting);
                }

            }
            else
            {
                if (stats.CurrentGameID != CurrentGameID)
                {
                    stats.CurrentGameID = CurrentGameID;
                    stats.GameTimestamp = CurrentGameTimestamp.SafeToEpochMilliseconds();
                    changes.Add(p => p.CurrentGameID);
                    changes.Add(p => p.GameTimestamp);
                }

                if (stats.GameState != CurrentGameState)
                {
                    stats.GameState = CurrentGameState;
                    changes.Add(p => p.GameState);
                }

                if (stats.GameStatusMessage != CurrentGameStatusMessage)
                {
                    stats.GameStatusMessage = CurrentGameStatusMessage;
                    changes.Add(p => p.GameStatusMessage);
                }

                if (stats.WatchingChannelID != WatchingChannelID)
                {
                    stats.WatchingChannelID = WatchingChannelID;
                    changes.Add(p => p.WatchingChannelID);
                }

                if (stats.BroadcastingGameID != BroadcastingGameID)
                {
                    stats.BroadcastingGameID = BroadcastingGameID;
                    changes.Add(p => p.BroadcastingGameID);
                }

                if (stats.IsBroadcasting != IsBroadcasting)
                {                    
                    stats.IsBroadcasting = IsBroadcasting;
                    changes.Add(p => p.IsBroadcasting);
                }                
            }

            if (tokenTimestamp > 0)
            {
                stats.TokenTimestamp = tokenTimestamp;
                changes.Add(p => p.TokenTimestamp);
            }

            if (stats.TwitchID != TwitchID)
            {
                stats.TwitchID = TwitchID;
                changes.Add(p => p.TwitchID);
            }

            if (stats.Capabilities != Capabilities)
            {
                stats.Capabilities = Capabilities;
                changes.Add(p => p.Capabilities);
            }

            if (changes.Any())
            {
                stats.Update(changes.ToArray());
            }
        }

        public UserPrivacySettings GetPrivacy()
        {
            return UserPrivacySettings.GetByUserOrDefault(UserID);         
        }

        public static bool CheckFriendshipPrivacy(User requestor, User target)
        {
            if (requestor == null || target == null)
            {
                return false;
            }

            // Check privacy
            var theirPrivacy = target.GetPrivacy();
            if (theirPrivacy.FriendRequestPrivacy == FriendRequestPrivacy.NoOne)
            {
                // Not accepting friend requests.
                return false;
            }
            if (theirPrivacy.FriendRequestPrivacy == FriendRequestPrivacy.FriendsOfFriends && Data.User.GetMutualFriends(requestor.UserID, target.UserID).Length == 0)
            {
                // No mutual friends
                return false;
            }
            return true;
        }

        public UserClientSettings GetClientSettings()
        {
            return UserClientSettings.GetByUserOrDefault(SourceConfiguration, UserID);
        }

        public UserContract ToNotification(ExternalAccount mergedAccount = null)
        {
            var externalCommunity = string.IsNullOrEmpty(WatchingChannelID) ? null : ExternalCommunity.GetByTwitchID(WatchingChannelID);
            var hints = FriendHint.GetAllLocal(p => p.UserID, UserID).Where(x => x.Type == FriendHintType.Game || x.Type == FriendHintType.Platform);

            if(mergedAccount == null && IsMerged)
            {
                mergedAccount = ExternalAccount.GetByTwitchUserID(TwitchID);
            }
            var tracking = UserTracking.GetLocal(UserID);

            return new UserContract
            {
                UserID = UserID,
                Username = Username,
                ConnectionStatus = ConnectionStatus,
                CurrentGameID = CurrentGameID,
                CurrentGameState = CurrentGameState,
                CurrentGameStatusMessage = CurrentGameStatusMessage,
                CurrentGameTimestamp = CurrentGameTimestamp,
                CustomStatusMessage = CustomStatusMessage,
                CustomStatusTimestamp = CustomStatusTimestamp,
                FriendCount = FriendCount,
                FriendMessagePushPreference = FriendMessagePushPreference,
                FriendRequestPushEnabled = FriendRequestPushEnabled,
                GroupMessagePushPreference = GroupMessagePushPreference,
                MentionsPushEnabled = MentionsPushEnabled,
                AvatarTimestamp = AvatarDate.ToEpochMilliseconds(),
                DisplayName = DisplayName,
                BroadcastingGameID = BroadcastingGameID,
                IsBroadcasting = IsBroadcasting,
                WatchingChannel = externalCommunity?.ToPublicContract(),
                PrivacySettings = GetPrivacy().ToContract(),
                ConfirmedFriendSyncCount = hints.Count(x => x.Status == FriendHintStatus.Normal),
                DeclinedFriendSyncCount = hints.Count(x => x.Status == FriendHintStatus.Deleted),

                IsPartner = mergedAccount?.IsPartnered ?? false,
                HasSeenPartnerUpsell = tracking?.HasSeenPartnerUpsell ?? false
            };
        }

        public UserProfileNotification ToProfileNotification(IEnumerable<MutualFriend> mutualFriends, IEnumerable<FriendHint> identities, Guid[] mutualGroupIDs)
        {
            return new UserProfileNotification
            {
                UserID = UserID,
                Username = Username,
                DisplayName = DisplayName,
                FriendCount = FriendCount,
                City = City,
                AboutMe = AboutMe,
                Name = Name,
                CountryCode = CountryCode,
                LastGameID = LastGameID,
                State = State,
                Identities = identities.Select(p => p.ToNotification()).ToArray(),
                MutualFriends = mutualFriends.Select(p => p.ToNotification()).ToArray(),
                MutualGroupIDs = mutualGroupIDs
            };
        }

        public UserProfileNotification GetUserProfile(int otherUserID = 0, bool includePlatformIdentities = false)
        {


            // Get the target user's friend hints
            var hints = FriendHint.GetAllLocal(p => p.UserID, UserID)
                .Where(
                    p =>
                        (
                            p.Type == FriendHintType.Game || p.Type == FriendHintType.Platform)  // Only game and platform identities
                            && p.Status == FriendHintStatus.Normal // Exclude anything that has been deleted
                            && (p.Visibility == FriendHintVisibility.VisibleToEveryone || p.Visibility == FriendHintVisibility.VisibleToFriends) // Only identities with proper visibility
                        )
                .ToArray();


            // Change out the display name to the appropriate public name
            foreach (var hint in hints)
            {
                hint.DisplayName = hint.GetPublicName();
            }

            
            // Exclude any platform identities for non-friends
            if (otherUserID > 0 && !includePlatformIdentities)
            {
                hints = hints.Where(p => p.Type != FriendHintType.Platform).ToArray();
            }

            MutualFriend[] mutualFriends;
            var mutualGroupIDs = new Guid[0];

            if (otherUserID > 0)
            {               
                mutualFriends = GetMutualFriendIDs(UserID, otherUserID);
                mutualGroupIDs = GetMutualGroupIDs(UserID, otherUserID);
            }
            else
            {
                mutualFriends = new MutualFriend[0];
            }
            
            return ToProfileNotification(mutualFriends, hints, mutualGroupIDs);
        }

        public void UpdateEmail(string newEmail)
        {
            EmailAddress = newEmail;
            Update(p => p.EmailAddress);
            
            var emailHints = FriendHint.GetAllLocal(h => h.UserID, UserID).Where(p => p.Type == FriendHintType.Email);
            
            foreach (var hint in emailHints)
            {
                hint.Delete();
            }

            var newHint = new FriendHint
            {
                Type = FriendHintType.Email,
                UserID = UserID,
                SearchTerm = newEmail,
            };
            newHint.InsertLocal();

            FriendHintSearchIndexer.CreateForHint(newHint);
        }

        public void ChangeUsername(string newUsername, string newDisplayName = null)
        {
            Logger.Info("Renaming user...", new { UserID, OldUsername = Username, NewUsername = newUsername, newDisplayName });

            LastNameChangeDate = DateTime.UtcNow;
            Username = newUsername;
            DisplayName = newDisplayName;
            Update(u => u.Username, u => u.DisplayName, u => u.LastNameChangeDate);
            UpdateStatistics();
            ReindexSearchHints();
           
            // Fix memberships, friendships, etc.
            AccountRenameWorker.Create(UserID, Username, DisplayName, this.GetTitleName());
        }

        public void ReindexSearchHints()
        {
            // Reindex user hints
            var existing = FriendHint.GetAllLocal(p => p.UserID, UserID);
            foreach (var hint in existing)
            {
                if (hint.Type == FriendHintType.Username || hint.Type == FriendHintType.Email)
                {
                    hint.Delete();
                }
            }

            var usernameHint = new FriendHint
            {
                UserID = UserID, 
                Type = FriendHintType.Username, 
                DisplayName = this.GetTitleName(), 
                SearchTerm = Username, 
                AvatarUrl = AvatarUrl, 
                Verification = FriendHintVerification.Verified,
                IsMerged = IsMerged
            };
            usernameHint.InsertLocal();
            FriendHintSearchIndexer.CreateForHint(usernameHint);

            if (!string.IsNullOrEmpty(EmailAddress))
            {
                var emailHint = new FriendHint
                {
                    Type = FriendHintType.Email,
                    UserID = UserID,
                    SearchTerm = EmailAddress,
                    IsMerged = IsMerged
                };
                emailHint.InsertLocal();
                FriendHintSearchIndexer.CreateForHint(emailHint);
            }  
        }

        public void UpdateTwitchID(string twitchID, ExternalAccount account)
        {
            if (TwitchID == twitchID)
            {
                return;
            }

            if (string.IsNullOrEmpty(twitchID))
            {
                var fallback = ExternalAccountMapping.GetAllLocal(a => a.UserID, UserID).FirstOrDefault(m => !m.IsDeleted && m.Type == AccountType.Twitch);
                twitchID = fallback == null ? string.Empty : fallback.ExternalID;
            }

            TwitchID = twitchID;
            Update(u => u.TwitchID);

            var stats = UserStatistics.GetByUserOrDefault(UserID, true);
            stats.TwitchID = twitchID;
            stats.EnsureEmotes(TimeSpan.Zero);
            stats.Update(s => s.TwitchID, s => s.EmoteMap, s => s.EmoteTimestamp);

            TwitchAccountResolver.UserLinkChanged(UserID, account);
        }

        public static User FindByUserID(int userID)
        {
            var userRegion = UserRegion.GetByUserID(userID);
            if (userRegion == null)
            {
                return null;
            }

            return userRegion.GetUser();
        }

        public bool ShouldSyncSocially()
        {
            // Synced recently enough          
            if (LastSocialSync != DateTime.MinValue && DateTime.UtcNow.Subtract(LastSocialSync) < SocialSyncInterval)
            {
                return false;

            }
                           
            return true;
        }

        public bool ShouldSyncExternalAccounts()
        {
            return (HasSyncedAccount || IsMerged) && !IsProvisional && (LastAccountSync == DateTime.MinValue || DateTime.UtcNow.Subtract(LastAccountSync) >= AccountSyncInterval);
        }


        public static IReadOnlyDictionary<int, User> GetUsersDictionary(HashSet<int> userIDs)
        {
            var regions = UserRegion.MultiGetLocal(userIDs.Select(id => new KeyInfo(id)));
            var users = regions.GroupBy(r => r.RegionID).SelectMany(g => MultiGet(g.Key, g.Select(r => new KeyInfo(r.UserID))));
            return users.ToDictionary(u => u.UserID);
        }

        public ExternalAccount GetTwitchAccount()
        {
            if (!IsMerged)
            {
                return null;
            }

            return ExternalAccount.GetByTwitchUserID(TwitchID);
        }

        public object GetLogData()
        {
            return new
            {
                UserID,
                Username,
                ConnectionStatus,
                DisplayName,                
                IsMerged,
                TwitchID,
                IsProvisional,
            };

        }
    }
}
