﻿using Curse.Aerospike;
using Curse.Extensions;
using System;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
   
    [TableDefinition(TableName ="UserBan", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class UserBan : BaseTable<UserBan>
    {
        [Column("UserID", KeyOrdinal = 1)]
        public int UserID { get; set; }

        [Column("Type")]
        public UserBanType Type { get; set; }

        [Column("StatusTime")]
        public long StatusTime { get; set; }

        [Column("Reason")]
        public string Reason { get; set; }

        public static bool IsBannedFrom(int userID, UserBanType type)
        {
            var ban = GetLocal(userID);
            return ban != null && ban.Type.HasFlag(type);
        }

        public static void BanUser(int userID, UserBanType type, bool notify = true)
        {
            var ban = GetLocal(userID);
            if (ban == null)
            {
                ban = new UserBan
                {
                    UserID = userID,
                    Type = type,
                    StatusTime = DateTime.UtcNow.ToEpochMilliseconds(),
                };
                ban.InsertLocal();

                if (notify)
                {
                    NotifyBanStatus(userID, ban.Type);
                }
            }
            else if (!ban.Type.HasFlag(type))
            {
                ban.Type |= type;
                ban.StatusTime = DateTime.UtcNow.ToEpochMilliseconds();
                ban.Update(p => p.Type, p => p.StatusTime);

                if (notify)
                {
                    NotifyBanStatus(userID, ban.Type);
                }
            }
        }

        public static void UnbanUser(int userID, UserBanType type, bool notify = true)
        {
            var ban = GetLocal(userID);
            if (ban == null)
            {
                return;
            }

            if (ban.Type.HasFlag(type))
            {
                ban.Type ^= type;
                ban.StatusTime = DateTime.UtcNow.ToEpochMilliseconds();
                ban.Reason = string.Empty;
                ban.Update(x => x.Type, x => x.StatusTime, x => x.Reason);

                if (notify)
                {
                    NotifyBanStatus(userID, ban.Type);
                }
            }
        }

        private static void NotifyBanStatus(int userID, UserBanType type)
        {
            var notification = new UserBanStatusNotification
            {
                UserID = userID,
                Bans = type
            };
            ClientEndpoint.DispatchNotification(userID, ep => UserBanStatusNotifier.Create(ep, notification));
        }
    }
}
