﻿using System;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using System.Diagnostics;

namespace Curse.Friends.Data.Models
{

    [TableDefinition(TableName = "UserBlock", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    [DebuggerDisplay("{UserID} -> {OtherUserID}: {Status}")]
    public class UserBlock : BaseTable<UserBlock>
    {
        /// <summary>
        /// The user ID of the person initiating the block
        /// </summary>
        [Column("UserID", KeyOrdinal = 1, IsIndexed = true)]
        public int UserID
        {
            get;
            set;
        }

        /// <summary>
        /// UserID of the other user
        /// </summary>
        [Column("OtherUserID", KeyOrdinal = 2, IsIndexed = true)]
        public int OtherUserID
        {
            get;
            set;
        }

        [Column("DateCreated")]
        public DateTime DateCreated { get; set; }

        [Column("DateModified")]
        public DateTime DateModified { get; set; }

        [Column("Status")]
        public UserBlockStatus Status { get; set; }

        public static UserBlock[] GetAllForUser(int userID, bool activeOnly = true)
        {
            var records = GetAllLocal(p => p.UserID, userID);

            if (activeOnly)
            {
                records = records.Where(p => p.Status == UserBlockStatus.Active).ToArray();
            }

            return records;
        }

        public static bool IsBlocked(int requestingUserID, int targetUserID)
        {
            if (requestingUserID == targetUserID)
            {
                throw new ArgumentException("Attempt to check block status of the same user.", nameof(requestingUserID));
            }
            // Check if the target user has blocked the requesting user
            var userBlock = GetLocal(targetUserID, requestingUserID);
            return userBlock != null && userBlock.Status == UserBlockStatus.Active;
        }

        public static UserBlock GetByUserIDAndOtherUserID(int requestingUserID, int otherUserID)
        {
            return GetLocal(requestingUserID, otherUserID);
        }

        public DateTime Recency
        {
            get
            {                
                return DateModified > DateTime.MinValue ? DateModified : DateCreated;
            }
        }

        public BlockedUserContract ToContract(UserStatistics statistics)
        {
            return new BlockedUserContract
            {
                UserID = OtherUserID,
                Username = statistics != null ? statistics.Username : "User-" + OtherUserID,
                DisplayName = statistics != null ? statistics.DisplayName : null,
            };
        }

        public static void ToggleBlock(User requestingUser, User otherUser, UserBlockStatus status)
        {
            var block = GetByUserIDAndOtherUserID(requestingUser.UserID, otherUser.UserID);


            if (block != null)
            {
                if (block.Status == status)
                {
                    return;
                }

                block.Status = status;
                block.DateModified = DateTime.UtcNow;
                block.Update(p => p.Status, p => p.DateModified);


            }
            else if (status == UserBlockStatus.Active)
            {
                block = new UserBlock
                {
                    UserID = requestingUser.UserID,
                    DateCreated = DateTime.UtcNow,
                    OtherUserID = otherUser.UserID,
                    Status = status
                };

                block.InsertLocal();
                DissolveFriendship(requestingUser, otherUser);
            }



            if (status == UserBlockStatus.Active)
            {
                DissolveFriendship(requestingUser, otherUser);

                var conversation = PrivateConversation.GetByUserIDAndOtherUserID(requestingUser.UserID, otherUser.UserID);
                if (conversation != null && !conversation.IsHidden)
                {
                    ((IConversationParent)conversation).ToggleHidden(true);
                }
            }
        }

        private static void DeclineSuggestions(FriendshipContext friendshipContext)
        {
            // Remove any friend suggestions, for either side
            var friendSuggestion = FriendSuggestion.Get(friendshipContext.MyRegion.RegionID, friendshipContext.MyRegion.UserID, friendshipContext.TheirRegion.UserID);

            if (friendSuggestion != null && friendSuggestion.Status == FriendSuggestionStatus.Pending)
            {
                friendSuggestion.Status = FriendSuggestionStatus.Declined;
                friendSuggestion.Update(p => p.Status);
                Logger.Debug("Auto declining friend suggestion", new { RequestingUser = friendshipContext.Me.Name, TargetUser = friendshipContext.Them.Name });
            }

            var theirFriendSuggestion = FriendSuggestion.Get(friendshipContext.TheirRegion.RegionID, friendshipContext.Them.UserID, friendshipContext.Me.UserID);

            if (theirFriendSuggestion != null && theirFriendSuggestion.Status == FriendSuggestionStatus.Pending)
            {
                theirFriendSuggestion.Status = FriendSuggestionStatus.Declined;
                theirFriendSuggestion.Update(p => p.Status);
                Logger.Debug("Auto declining friend suggestion", new { RequestingUser = friendshipContext.Me.Name, TargetUser = friendshipContext.Them.Name });
            }

        }

        private static void DissolveFriendship(User requestingUser, User targetUser)
        {

            // Ensure that the friendship exists            
            var friendshipContext = new FriendshipContext(requestingUser.UserID, targetUser.UserID);


            // Decline any suggestions
            DeclineSuggestions(friendshipContext);
           
            // Remove friendship, if it exists
            var myFriendship = friendshipContext.GetMyFriendship();
            var theirFriendship = friendshipContext.GetTheirFriendship();

            // If neither side of the friendship exists, this is an invalid request
            if (myFriendship == null || theirFriendship == null)
            {
                return;
            }

            var wasSurfaced = myFriendship.IsSurfaced;

            // Remove the friendship
            Friendship.Remove(false, friendshipContext.Me, myFriendship, friendshipContext.MyRegion, friendshipContext.Them, theirFriendship, friendshipContext.TheirRegion);
            Logger.Debug("Removing friendship", new { RequestingUser = requestingUser.Name, TargetUser = targetUser.Name });

            // Only send a notification if the friendship was surfaced in the client in some fashion
            if (wasSurfaced)
            {
                // Notify Myself
                new FriendshipRemovedResolver { UserID = friendshipContext.Me.UserID, FriendID = friendshipContext.Them.UserID }.Enqueue();

                // Notify Them
                new FriendshipRemovedResolver { UserID = friendshipContext.Them.UserID, FriendID = friendshipContext.Me.UserID }.Enqueue();
            }
        }
        
    }
}
