﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "UserClientSettings", KeySpace = "CurseVoice", ReplicationMode = ReplicationMode.None)]
    public class UserClientSettings : BaseTable<UserClientSettings>
    {
        [Column("UserID", KeyOrdinal = 1)]
        public int UserID { get; set; }

        [Column("DateUpdated")]
        public DateTime DateUpdated { get; set; }

        [Column("Global")]
        public string GlobalSettingsBlob { get; set; }

        [Column("Desktop")]
        public string DesktopSettingsBlob { get; set; }

        [Column("Web")]
        public string WebSettingsBlob { get; set; }

        [Column("Mobile")]
        public string MobileSettingsBlob { get; set; }

        public static UserClientSettings GetByUserOrDefault(AerospikeConfiguration configuration, int userId)
        {
            var settings = Get(configuration, userId);
            if (settings != null)
            {
                return settings;
            }
         
            settings = new UserClientSettings
            {
                UserID = userId,
                DateUpdated = DateTime.UtcNow,
            };
            settings.Insert(configuration);

            return settings;
        }

        public UserClientSettingsNotification ToNotification()
        {
            return new UserClientSettingsNotification
            {
                UserID = UserID,
                DateUpdated = DateUpdated.ToEpochMilliseconds(),
                DesktopSettings = DesktopSettingsBlob,
                GlobalSettings = GlobalSettingsBlob,
                WebSettings = WebSettingsBlob,
                MobileSettings = MobileSettingsBlob
            };
        }

        public void ChangeSettings(string globalSettings, string desktopSettings, string webSettings, string mobileSettings)
        {
            var updates = new List<Expression<Func<UserClientSettings, object>>>();

            if (globalSettings != null && globalSettings != GlobalSettingsBlob)
            {
                GlobalSettingsBlob = globalSettings;
                updates.Add(s => s.GlobalSettingsBlob);
            }

            if (desktopSettings != null && desktopSettings != DesktopSettingsBlob)
            {
                DesktopSettingsBlob = desktopSettings;
                updates.Add(s => s.DesktopSettingsBlob);
            }

            if (webSettings != null && webSettings != WebSettingsBlob)
            {
                WebSettingsBlob = webSettings;
                updates.Add(s => s.WebSettingsBlob);
            }

            if (mobileSettings != null && mobileSettings != MobileSettingsBlob)
            {
                MobileSettingsBlob = mobileSettings;
                updates.Add(s => s.MobileSettingsBlob);
            }

            if (updates.Any())
            {
                DateUpdated = DateTime.UtcNow;
                updates.Add(s => s.DateUpdated);

                Update(updates.ToArray());

                var notification = ToNotification();
                ClientEndpoint.DispatchNotification(UserID, ep => UserClientSettingsNotifier.Create(ep, notification));
            }
        }
    }
}
