﻿using System;
using System.Collections.Generic;
using System.Linq;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Extensions;

namespace Curse.Friends.Data.Models
{
    [TableDefinition(TableName = "UserPrivateConversationThrottle", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class UserPrivateConversationThrottle : BaseTable<UserPrivateConversationThrottle>
    {
        [Column("UserID", KeyOrdinal = 1)]
        public int UserID { get; set; }

        /// <summary>
        /// The recent history of private conversations timestamps where this user sent a message to a stranger
        /// </summary>
        [Column("History")]
        public Dictionary<string, long> History { get; set; }

        [Column("BannedUntil")]
        public long BannedUntil { get; set; }

        [Column("Timestamps")]
        public long[] Timestamps { get; set; }

        public void Track(string conversationID, long timestamp, int dailyLimit)
        {
            // Get or create the dictionary
            var dict = History ?? new Dictionary<string, long>();

            var oneDayAgo = DateTime.UtcNow.AddDays(-1).ToEpochMilliseconds();
            
            // Remove anything older than 24 hours
            var newDictionary = dict.Where(pair => pair.Value >= oneDayAgo)
                                     .ToDictionary(pair => pair.Key, pair => pair.Value);

            // Track this
            newDictionary[conversationID] = timestamp;

            // Update record's dictionary
            History = newDictionary;

            // Count the number of conversations in the last day
            if (newDictionary.Count > dailyLimit)
            {
                BannedUntil = DateTime.UtcNow.AddDays(1).ToEpochMilliseconds();
                Logger.Info("Banning user from stranger PMs as they have contacted more than " + dailyLimit + " strangers in 24 hours.", new {UserID});
                Update(p => p.History, p => p.BannedUntil);
            }
            else
            {
                Update(p => p.History);
            }            
        }

        public static UserPrivateConversationThrottle GetOrCreateForUser(int userID)
        {
            var throttle = GetLocal(userID);

            if (throttle == null)
            {
                throttle = new UserPrivateConversationThrottle
                {
                    UserID = userID,
                    History = new Dictionary<string, long>(),
                    Timestamps = new long[0]
                };
                throttle.InsertLocal();
            }

            return throttle;
        }

        public bool ShouldThrottleForFrequency(int maxPerSecond, int maxPerMinute, long timestamp, out long? retryAfter)
        {
            var timestamps = new List<long>(Timestamps ?? new long[0]);
            const int secondMillis = 1000;
            const int minuteMillis = 60000;

            if(maxPerSecond == 0 || maxPerMinute == 0)
            {
                retryAfter = 0;
                return false;
            }

            var attempt = 1;
            do
            {
                if (attempt > 1)
                {
                    Refresh();
                    timestamps = Timestamps.ToList();
                }

                // 1-second throttle
                var secondTimestamps = timestamps.Where(t => t > timestamp - secondMillis).ToArray();
                if (secondTimestamps.Length >= maxPerSecond)
                {
                    retryAfter = secondTimestamps.Min() + secondMillis - timestamp;
                    return true;
                }

                // 1-minute throttle
                var minuteTimestamps = timestamps.Where(t => t > timestamp - minuteMillis).ToArray();
                if (minuteTimestamps.Length >= maxPerMinute)
                {
                    retryAfter = minuteTimestamps.Min() + minuteMillis - timestamp;
                    return true;
                }

                timestamps.Add(timestamp);
                Timestamps = timestamps.Where(t=>t > timestamp - minuteMillis).ToArray();

                try
                {
                    // This can be bypassed by running in multiple regions, but picking a home region could slow down processing
                    Update(UpdateMode.Concurrent, LocalConfiguration, t => t.Timestamps);
                    retryAfter = 0;
                    return false;
                }
                catch (AerospikeException)
                {
                    attempt++;
                }
            } while (attempt < 4);

            Logger.Warn("Failed to write throttle timestamps to DB, assuming spam in current region", new { UserID = UserID, Timestamps = Timestamps });
            retryAfter = 0;
            return true;
        }

    }
}
