﻿using System;
using Curse.Aerospike;
using Curse.Extensions;

namespace Curse.Friends.Data
{

    [TableDefinition(TableName = "UserRegion", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class UserRegion : BaseTable<UserRegion>
    {        
        [Column("UserID", KeyOrdinal = 1)]
        public int UserID
        {
            get;
            set;

        }
        
        [Column("RegionID")]
        public int RegionID
        {
            get;
            set;

        }

        [Column("IsRelocating")]
        public bool IsRelocating
        {
            get;
            set;
        }

        [Column("ReloTimestamp")]
        public long RelocationTimestamp
        {
            get;
            set;
        }

        public User GetUser()
        {
            return User.Get(RegionID, UserID);
        }

        public static UserRegion GetByUserID(int userID, bool searchRemotely = true, bool logWarning = true)
        {
            var userRegion = GetLocal(userID);
            if (userRegion != null)
            {
                return userRegion;
            }

            if (searchRemotely)
            {
                foreach (var region in RemoteConfigurations)
                {
                    userRegion = Get(region, userID);
                    if (userRegion != null)
                    {
                        Logger.Debug("Found user region in remote database.", userRegion);
                        userRegion.InsertLocal();
                        return userRegion;
                    }
                }
            }

            if (logWarning)
            {
                Logger.Warn("Unable to find region for user.", new { userID });
            }            

            return null;
        }

        public bool WasRecentlyRelocated()
        {
            var reloDate = RelocationTimestamp.FromEpochMilliconds();
            return !IsRelocating && DateTime.UtcNow.Subtract(reloDate) < TimeSpan.FromDays(1);
        }

        public bool IsRecentlyRelocating()
        {
            if (!IsRelocating)
            {
                return false;

            }

            if (RelocationTimestamp == 0)
            {
                return true;
            }

            var reloDate = RelocationTimestamp.FromEpochMilliconds();
            return DateTime.UtcNow.Subtract(reloDate) < TimeSpan.FromMinutes(1);
        }

        public bool ShouldRelocate(out string reason)
        {            
            // If the current region is already the local region, nothing to do
            if(RegionID == LocalConfigID)
            {
                reason = "Already in local region.";
                return false;
            }

            // If a relocation is in flight, determine how old it is
            if (IsRecentlyRelocating())
            {
                reason = "User relocation is likely in flight.";
                return false;
            }

            // If a relocation happened in the last day, do not relocate again
            if (WasRecentlyRelocated())
            {
                reason = "User was recently relocated.";
                return false;
            }

            reason = "User should be relocated!";
            return true;
        }
    }
}
