﻿using System;
using Curse.Aerospike;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;

namespace Curse.Friends.Data
{

    [TableDefinition(TableName = "UserStatistics", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class UserStatistics : BaseTable<UserStatistics>, IUserIdentity
    {                
        [Column("UserID", KeyOrdinal = 1)]
        public int UserID
        {
            get;
            set;

        }

        [Column("Username")]
        public string Username
        {
            get;
            set;
        }

        [Column("DisplayName")]
        public string DisplayName
        {
            get;
            set;
        }

        string IUserIdentity.Nickname
        {
            get { return null; }
        }

        [Column("ConnStatus")]
        public UserConnectionStatus ConnectionStatus
        {
            get;
            set;
        }


        [Column("StatusTimes")]
        public long ConnectStatusTimestamp
        {
            get;
            set;
        }

        [Column("StatusMessage")]
        public string StatusMessage
        {
            get;
            set;
        }

        [Column("DateLastSeen")]
        public long DateLastSeen
        {
            get;
            set;
        }        
        
        public int FriendCount
        {
            get
            {
                return FriendIDs != null ? FriendIDs.Count : 0;
            }            
        }

        [Column("FriendIDs")]
        public HashSet<int> FriendIDs
        {
            get;
            set;
        }

        [Column("IsCasting")]
        public bool IsBroadcasting
        {
            get;
            set;
        }

        [Column("CastingGameID")]
        public int BroadcastingGameID
        {
            get;
            set;
        }

        [Column("WatchChanID")]
        public string WatchingChannelID
        {
            get;
            set;
        }

        [Column("CurGameID")]
        public int CurrentGameID
        {
            get;
            set;
        }

        [Column("GameState")]
        public int GameState
        {
            get;
            set;
        }

        [Column("GameStatus")]
        public string GameStatusMessage
        {
            get;
            set;
        }

        [Column("GameTimestamp")]
        public long GameTimestamp
        {
            get;
            set;
        }

        [Column("AvatarTime")]
        public long AvatarTimestamp
        {
            get;
            set;
        }

        [Column("TokenTime")]
        public long TokenTimestamp
        {
            get; 
            set;
        }

        [Column("EmoteMap")]
        public Dictionary<string, int> EmoteMap { get; set; }

        [Column("EmoteTime")]
        public long EmoteTimestamp { get; set; }

        [Column("TwitchID")]
        public string TwitchID { get; set; }

        [Column("Capabilities")]
        public ClientCapability Capabilities { get; set; }

        public bool HasTwitchID
        {
            get { return !string.IsNullOrEmpty(TwitchID); }
        }

        public static UserStatistics GetByUserID(int userID)
        {
            return GetLocal(userID);            
        }

        public static UserStatistics GetByUserOrDefault(int userID, bool insertIfMissing = false)
        {
            var userStats = GetLocal(userID);
            if (userStats == null)
            {
                userStats = new UserStatistics {UserID = userID, FriendIDs = new HashSet<int>()};

                if (insertIfMissing)
                {
                    userStats.InsertLocal();
                }
            }

            return userStats;
        }

        public static UserStatistics GetByUser(int userID)
        {
            return GetLocal(userID);
        }

        public static IReadOnlyCollection<UserStatistics> GetAllByUserIDs(IEnumerable<int> userIDs)
        {
            return MultiGetLocal(userIDs.Select(p => new KeyInfo(p)));
        }

        public static Dictionary<int, UserStatistics> GetDictionaryByUserIDs(IEnumerable<int> userIDs)
        {
            var stats = MultiGetLocal(userIDs.Select(p => new KeyInfo(p)));
            return stats.ToDictionary(p => p.UserID);
        }

        public void SetMinimumTokenDate(DateTime timestamp)
        {
            var epoch = timestamp.ToEpochMilliseconds();
            if (epoch > TokenTimestamp)
            {
                TokenTimestamp = epoch;
                Update(p => p.TokenTimestamp);
            }
        }

        public void EnsureEmotes(TimeSpan throttleLimit)
        {
            if (EmoteMap != null && (DateTime.UtcNow - throttleLimit).ToEpochMilliseconds() < EmoteTimestamp)
            {
                return;
            }

            var twitchIdentities = new HashSet<string>(
                ExternalAccountMapping.GetAllLocal(m => m.UserID, UserID)
                    .Where(m => !m.IsDeleted && m.Type == AccountType.Twitch)
                    .Select(m => m.ExternalID));

            TwitchEmote[] twitchEmotes;
            if (twitchIdentities.Count > 0)
            {
                // Sync from Twitch
                var accounts = ExternalAccount.MultiGetLocal(twitchIdentities.Select(id => new KeyInfo(id, AccountType.Twitch)));
                var emoteIDs = new HashSet<long>();
                foreach (var account in accounts)
                {
                    var emotesResponse = TwitchApiHelper.Default.GetUserEmotes(account.ExternalID, account.AuthToken);
                    if (emotesResponse.Status != TwitchResponseStatus.Success)
                    {
                        continue;
                    }

                    if (emotesResponse.Value.EmoticonSets == null)
                    {
                        continue;
                    }

                    foreach (var emoteSet in emotesResponse.Value.EmoticonSets)
                    {
                        foreach (var emote in emoteSet.Value)
                        {
                            emoteIDs.Add(emote.ID);
                        }
                    }
                }

                if(emoteIDs.Count == 0 && EmoteMap != null)
                {
                    // Don't overwrite emotes with no emotes
                    return;
                }


                twitchEmotes = emoteIDs.Count > 0 ? TwitchEmote.MultiGetLocal(emoteIDs.Select(id => new KeyInfo(id))).ToArray() : TwitchEmote.GetAllLocal(e => e.EmoticonSet, 0);

                if(FriendsServiceConfiguration.Instance.AdditionalEmoteIDs.Length > 0)
                {
                    twitchEmotes = twitchEmotes.Concat(TwitchEmote.MultiGetLocal(FriendsServiceConfiguration.Instance.AdditionalEmoteIDs.Select(id => new KeyInfo(id)))).ToArray();
                }
            }
            else
            {
                twitchEmotes = new TwitchEmote[0];
            }

            var emoteMap = new Dictionary<string, int>();
            if (twitchEmotes.Length > 0)
            {
                foreach (var emote in twitchEmotes.OrderBy(e => e.EmoticonSet))
                {
                    emoteMap[emote.Regex] = (int) emote.EmoteID;
                }
            }
            EmoteMap = emoteMap;
            EmoteTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
            Update(u => u.EmoteTimestamp, u => u.EmoteMap);
        }

        public static UserStatistics[] MultiGetAllForUserIDs(int[] allGroupMemberIDs)
        {
            return MultiGetLocal(allGroupMemberIDs.Select(id => new KeyInfo(id))).ToArray();
        }

        public ExternalCommunity GetWatchingCommunity()
        {
            if (string.IsNullOrEmpty(WatchingChannelID))
            {
                return null;
            }

            return ExternalCommunity.GetByTwitchID(WatchingChannelID);
        }

        public static void UpdateUserStatistics(IEnumerable<int> userIDs)
        {
            if (userIDs == null)
            {
                return;
            }

            var ids = new HashSet<int>(userIDs);
            var users = User.GetUsersDictionary(ids);

            foreach (var user in users)
            {
                user.Value.UpdateStatistics();
            }
        }
    }
}
