﻿using System;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Utils;

namespace Curse.Friends.Data.Models
{
    public enum VanityUrlType
    {
        Group,


        Blacklist = 1000,
        Reserved = 1001
    }

    /// <summary>
    /// This does not implement IModelRegion so it throws an exception if InsertLocal is used instead of the defined Insert method
    /// </summary>
    [TableDefinition(TableName = "VanityUrl", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.HomeRegion)]
    public class VanityUrl : BaseTable<VanityUrl>
    {
        public const int MaxUrlLength = 64;

        [Column("Url", KeyOrdinal = 1)]
        public string Url { get; set; }

        [Column("DisplayUrl")]
        public string DisplayUrl { get; set; }

        [Column("Type")]
        public VanityUrlType Type { get; set; }
        
        /// <summary>
        /// A nullable value that indicates that this vanity URL should redirect to another url
        /// </summary>
        [Column("MappedUrl")]
        public string MappedUrl { get; set; }

        [Column("MappedID")]
        public string MappedID { get; set; }

        /// <summary>
        /// The region ID used as the source of truth for coordinating concurrent access
        /// </summary>
        private const int _regionID = 1;

        /// <summary>
        /// All inserts should ultimately go through this method to ensure they go to the right region
        /// </summary>
        private void Insert()
        {
            Insert(_regionID, UpdateMode.Concurrent);
        }

        public static bool Exists(string url, bool allowDirtyRead = true)
        {
            return GetByUrl(url, allowDirtyRead) != null;

        }

        public static string FindNextAvailable(string initialUrl)
        {
            // Now find a custom URL
            var url = initialUrl;
            var counter = 0;
            while (Exists(url))
            {
                if (counter == 100)
                {                    
                    return null;                    
                }

                url = url + (++counter);
            }

            return url;
        }

        public static VanityUrl GetByUrl(string url, bool allowDirtyRead = true)
        {
            if (!url.StartsWith("/"))
            {
                url = "/" + url;
            }

            url = url.ToLowerInvariant();

            return allowDirtyRead ? GetLocal(url) : Get(_regionID, url);            
        }

        public static VanityUrl CreateForGroup(string url, Guid groupID)
        {
            if (!url.StartsWith("/"))
            {
                url = "/" + url;
            }

            var vanityUrl = new VanityUrl
            {
                Url = url.ToLowerInvariant(),
                DisplayUrl = url,
                MappedID = groupID.ToString(),
                Type = VanityUrlType.Group
            };

            vanityUrl.Insert();

            return vanityUrl;
        }

        public static VanityUrl RandomForGroup(Guid groupID)
        {
            var url = FindNextAvailableCode("/servers");
            
            // This is hard coded to region 1 for all writes, to avoid concurrency issues.
            var vanityUrl = new VanityUrl
            {
                Url = url.ToLowerInvariant(),
                DisplayUrl = url,
                MappedID = groupID.ToString(),
                Type = VanityUrlType.Group
            };

            vanityUrl.Insert();

            return vanityUrl;
        }

        private const int MaxCodeAttempts = 20;
        private const int MinCodeLength = 6;
        private const int MaxCodeLength = 12;

        public static string FindNextAvailableCode(string prefix = "")
        {
            var attempts = 0;
            var currentSize = MinCodeLength;

            while (true)
            {
                if (++attempts > MaxCodeAttempts)
                {
                    throw new Exception("Unable to find a suitable url after " + attempts + " attempts. This should be impossible...");
                }

                var url = prefix + '/' + ShortCodeGenerator.RandomCharacters(currentSize);

                // If the url doesn't exist, we're g2g!
                if (!Exists(url))
                {
                    return url;
                }

                // Keep growing the size of the url, until we fine one
                if (currentSize < MaxCodeLength)
                {
                    currentSize = currentSize + (attempts / 10);    
                }                
            }
        }

        public static string GetGroupIDFromUrl(string url)
        {
            if (!url.StartsWith("/"))
            {
                url = "/" + url;
            }

            url = url.ToLowerInvariant();
            var vanityUrl = GetLocal(url);

            if (vanityUrl == null || vanityUrl.Type != VanityUrlType.Group)
            {
                return null;
            }

            return vanityUrl.MappedID;
        }

        protected override void Validate()
        {
            if (!Url.SafeRange(1, MaxUrlLength))
            {
                throw new Exception("Invalid Url: " + Url);
            }

            if (!Url.ToLowerInvariant().Equals(Url))
            {
                throw new Exception("Invalid Url: " + Url);
            }

            if (!Url.StartsWith("/"))
            {
                throw new Exception("Invalid Url: " + Url);
            }

            Guid groupID;
            if (Type == VanityUrlType.Group && !Guid.TryParse(MappedID, out groupID))
            {
                throw new Exception("Invalud mapped ID: " + MappedID);
            }
        }

        public static VanityUrl BlackList(string url)
        {
            var vanity = GetByUrl(url);
            if (vanity == null)
            {
                vanity = new VanityUrl
                {
                    Type = VanityUrlType.Blacklist,
                    Url = url.ToLowerInvariant(),
                    DisplayUrl = url,
                };
                vanity.Insert();
            }
            else
            {

                switch (vanity.Type)
                {
                    case VanityUrlType.Blacklist:
                        break;
                    case VanityUrlType.Reserved:
                        vanity.Type = VanityUrlType.Blacklist;
                        vanity.Update(v => v.Type);
                        break;
                    case VanityUrlType.Group:
                        Logger.Warn("Blacklisting url already in use by a group!");
                        vanity.Type = VanityUrlType.Blacklist;
                        vanity.MappedID = string.Empty;
                        vanity.MappedUrl = string.Empty;
                        vanity.Update();

                        // TODO: Change group's vanity URL
                        break;
                    default:
                        Logger.Warn("Unexpected vanity url type during blacklist, performing blacklist but affected entities may not be updated", vanity);
                        vanity.Type = VanityUrlType.Blacklist;
                        vanity.MappedID = string.Empty;
                        vanity.MappedUrl = string.Empty;
                        vanity.Update();
                        break;
                }
            }

            return vanity;
        }

        public static VanityUrl Reserve(string url)
        {
            var vanity = GetByUrl(url);
            if (vanity == null)
            {
                vanity = new VanityUrl
                {
                    Type = VanityUrlType.Reserved,
                    Url = url.ToLowerInvariant(),
                    DisplayUrl = url,
                };
                vanity.Insert();
            }
            else
            {

                switch (vanity.Type)
                {
                    case VanityUrlType.Reserved:
                        break;
                    case VanityUrlType.Group:
                        Logger.Warn("Reserving url already in use by a group", vanity);
                        vanity.Type = VanityUrlType.Reserved;
                        vanity.MappedID = string.Empty;
                        vanity.MappedUrl = string.Empty;
                        vanity.Update();

                        // TODO: Change group's vanity URL
                        break;
                    case VanityUrlType.Blacklist:
                        Logger.Warn("Reserving a previously blacklisted url");
                        vanity.Type = VanityUrlType.Reserved;
                        vanity.Update(v => v.Type);
                        break;
                    default:
                        Logger.Warn("Unexpected vanity url type during reserve, performing reserve but affected entities may not be updated", vanity);
                        vanity.Type = VanityUrlType.Reserved;
                        vanity.MappedID = string.Empty;
                        vanity.MappedUrl = string.Empty;
                        vanity.Update();
                        break;
                }
            }

            return vanity;
        }
    }
}
