﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudQueue;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.Data
{
    public class AccountRenameWorker : BaseCloudQueueWorkerMessage<AccountRenameWorker>
    {
        public int UserID { get; set; }
        public string Username { get; set; }
        public string DisplayName { get; set; }
        public string FormattedDisplayName { get; set; }

        public static void Create(int userID, string username, string displayName, string formattedDisplayName)
        {
            new AccountRenameWorker
            {
                UserID = userID,
                Username = username,
                DisplayName = displayName,
                FormattedDisplayName = formattedDisplayName
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(AccountRenameWorker worker)
        {
            if (string.IsNullOrEmpty(worker.Username))
            {
                Logger.Warn("Unable to process rename event. The username is null or empty!", new { worker });
                return;
            }

            if (string.IsNullOrEmpty(worker.FormattedDisplayName))
            {
                Logger.Debug("The formatted display name was not supplied. Falling back to the username.", new { worker });
                worker.FormattedDisplayName = worker.Username;
            }

            Logger.Debug("Account rename started: " + worker.Username);

            var memberships = GroupMember.GetAllByUserID(worker.UserID);
            Logger.Debug("Processing group memberships...", new { worker, memberships = memberships.Length });

            //Groups
            foreach (var member in memberships.Where(p => p.IsRootGroup))
            {
                var group = Group.GetByID(member.GroupID);
                if (group == null)
                {
                    Logger.Warn("Unable to locate a group: " + member.GroupID);
                    continue;
                }

                // Get the member from the home region
                var regionalMember = group.GetMember(member.UserID);
                regionalMember.Username = worker.Username;
                regionalMember.DisplayName = worker.DisplayName;
                regionalMember.Update(p => p.Username, p => p.DisplayName);


                // Legacy PMs
                var privateMessageConversation = GroupPrivateConversation.GetAllByGroupIDAndUserID(group.GroupID, member.UserID);
                foreach (var convo in privateMessageConversation)
                {
                    var otherSide = GroupPrivateConversation.GetBySenderIDAndRecipientIDAndGroupID(convo.OtherUserID, convo.UserID, group.GroupID);
                    if (otherSide != null)
                    {
                        otherSide.OtherUsername = worker.Username;                        
                    }
                }

                GroupMemberIndexWorker.CreateMemberNicknameWorker(group, worker.UserID, regionalMember.Username, regionalMember.Nickname, regionalMember.DisplayName);
                GroupChangeCoordinator.UpdateUsers(group, member.UserID, new HashSet<int> { member.UserID });
            }

            // Friends
            foreach (var region in Friendship.AllConfigurations)
            {
                Logger.Debug("Processing friends memberships in region: " + region.RegionKey, new { worker.UserID, worker, memberships = memberships.Length });

                var friends = Friendship.GetAll(region, p => p.OtherUserID, worker.UserID);
                foreach (var friend in friends)
                {
                    // If nickname specified is username, clear it out
                    if (friend.OtherUserNickname != null && friend.OtherUserNickname.Equals(friend.OtherUsername, StringComparison.InvariantCultureIgnoreCase))
                    {                        
                        friend.OtherUserNickname = string.Empty;
                    }
                    friend.OtherUserDisplayName = worker.DisplayName;
                    friend.OtherUsername = worker.Username;
                    friend.Update(p => p.OtherUserNickname, p => p.OtherUsername, p => p.OtherUserDisplayName);
                }
            }

            // Private Conversations
            var privateConversations = PrivateConversation.GetAllLocal(p => p.OtherUserID, worker.UserID);

            foreach (var conversation in privateConversations)
            {
                conversation.Title = worker.FormattedDisplayName;                
                conversation.Update(p => p.Title);
            }

            RegionalUserChangeResolver.Create(worker.UserID);

            Logger.Debug("Account rename completed: " + worker.Username);
        }
    }
}
