﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.Data
{
    /// <summary>
    /// Responsible to update all connected ClientEndPoints of a userID login status changes
    /// </summary>
    public class ClientEndpointResolver : BaseCloudQueueWorkerMessage<ClientEndpointResolver>
    {
        public ClientEndpointResolver() { }

        public int UserID { get; set; }

        public int UserRegionID { get; set; }

        /// <summary>
        /// Creates and queues the notification to be processed by workerServer.
        /// </summary>
        /// <param name="userID"></param>
        /// <param name="userRegionID"></param>
        public static void Create(int userID, int userRegionID)
        {
            try
            {
                new ClientEndpointResolver { UserID = userID, UserRegionID = userRegionID }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create!");
            }
        }

        public static void StartProcessor()
        {
            StartProcessor(ProcessMessage);
        }

        /// <summary>
        /// WorkerServer dequeues and notifies all the active groups that user belongs to.
        /// </summary>
        /// <param name="value"></param>
        private static void ProcessMessage(ClientEndpointResolver value)
        {

            // Get all of this user's root group memberships
            var memberships = GroupMember.GetAllByUserID(value.UserID)
                                         .Where(p => p.GroupID == p.RootGroupID 
                                             && !p.IsDeleted)
                                         .ToArray();

            if (!memberships.Any())
            {
                return;
            }

            var activeGroups = Group.MultiGetLocal(memberships.Select(p => new KeyInfo(p.GroupID)))
                                    .Where(p => !string.IsNullOrEmpty(p.MachineName))
                                    .ToArray();

            if (!activeGroups.Any())
            {
                return;
            }

            // Get the latest snapshot of all connected endpoints for this user
            var endpoints = ClientEndpoint.GetAllDeliverable(value.UserID).ToArray();

            foreach (var group in activeGroups)
            {
                if (QueueConfiguration.SingleRegionMode)
                {
                    if (group.RegionID != Group.LocalConfigID)
                    {
                        Logger.Info("Remote group region detected in single region mode. It is being moved to the local region!");
                        group.RegionID = Group.LocalConfigID;
                        group.MachineName = string.Empty;
                        group.Update(p => p.RegionID, p => p.MachineName);

                        GroupHostManager.EnsureServiceHost(group);
                    }
                }

                try
                {
                    new GroupEndpointCoordinator(group) { UserID = value.UserID, ConnectedEndpoints = endpoints }.Enqueue();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to enqueue GroupEndpointCoordinator", new { group.GroupID, group.RegionID, group.MachineName });
                }

            }

        }
    }
}
