﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.CloudQueue;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.Data
{
    public enum ConversationMessageBulkWorkerType
    {
        DeleteMessages
    }

    public class ConversationMessageBulkWorker : BaseCloudQueueShoveledMessage<ConversationMessageBulkWorker>
    {
        public ConversationMessageBulkWorkerType Type { get; set; }

        public string ConversationID { get; set; }

        public int RequestorUserID { get; set; }

        public string RequestorUsername { get; set; }

        public DateTime RequestDate { get; set; }

        public int SenderUserID { get; set; }

        public DateTime StartDate { get; set; }

        public static void CreateBulkDelete(string conversationID, int requestorID, string requestorUsername, DateTime requestDate, int senderID, DateTime startDate)
        {
            new ConversationMessageBulkWorker
            {
                Type = ConversationMessageBulkWorkerType.DeleteMessages,
                RequestorUserID = requestorID,
                RequestorUsername = requestorUsername,
                ConversationID = conversationID,
                RequestDate = requestDate,
                StartDate = startDate,
                SenderUserID = senderID,
                SourceRegionID = QueueConfiguration.LocalRegionID
            }.Enqueue();
        }
        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(ConversationMessageBulkWorker worker)
        {
            try
            {
                switch (worker.Type)
                {
                    case ConversationMessageBulkWorkerType.DeleteMessages:
                        BulkDeleteMessages(worker);
                        break;
                    default:
                        Logger.Error("Unknown ConversationMessageBulkWorkerType: " + worker.Type);
                        break;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process ConversationMessageBulkWorker", worker);
            }
        }

        private static void BulkDeleteMessages(ConversationMessageBulkWorker worker)
        {
            Logger.Debug("Starting bulk delete worker", worker);

            Logger.Debug("Getting messages to bulk delete", worker);
            var messagesToNotify = new List<ConversationMessage>();
            foreach (var messageResult in ConversationManager.ScrollByRootConversationAndUser(worker.ConversationID, worker.SenderUserID, worker.StartDate, worker.RequestDate))
            {
                var messages = messageResult.ToArray();
                var messagesToAdd = 10 - messagesToNotify.Count;
                messagesToNotify.AddRange(messages.Take(messagesToAdd));

                ConversationManager.BulkDelete(messages, worker.RequestorUserID, worker.RequestorUsername, worker.RequestDate);
            }            

            // Only send notifications from the source region to prevent extra traffic
            if (worker.SourceRegionID == QueueConfiguration.LocalRegionID)
            {
                var conversationContainer = ConversationManager.GetConversationContainer(worker.RequestorUserID, worker.ConversationID);
                if (conversationContainer == null)
                {
                    Logger.Warn("Unable to notify users of bulk deleted messages, conversation container is null", worker);
                }
                else
                {
                    Logger.Debug("Notifying users of recent deleted messages", worker);
                    if (messagesToNotify.Any())
                    {
                        foreach (var messageToNotify in messagesToNotify)
                        {
                            conversationContainer.OnChatMessageChanged(messageToNotify, ConversationNotificationType.Deleted);
                        }

                        SendBulkMessageDeleteNotification(worker, conversationContainer);
                    }
                }
            }
            Logger.Debug("Finished bulk delete worker", worker);
        }

        private static void SendBulkMessageDeleteNotification(ConversationMessageBulkWorker worker, IConversationContainer conversationContainer)
        {
            var group = conversationContainer as Group; 
            if(group == null)
            {
                return; 
            }
            
            GroupBulkMessageDeleteCoordinator.Create(group, worker.SenderUserID, worker.StartDate, worker.RequestDate, worker.RequestorUserID, worker.RequestorUsername);
        }
    }
}
