﻿using Curse.CloudQueue;
using Newtonsoft.Json;
using System;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;

namespace Curse.Friends.Data.Queues
{

    public enum ConversationMessageWorkerType
    {
        NewMesssage,
        EditMessage,
        DeleteMessage,
        EditAttachment,
        DeleteAttachment,
        LikeMessage,
        UpdateMessageLikes
    }

    [CloudQueueProcessor(4, true)]
    [CloudQueue(true)]
    public class ConversationMessageWorker : BaseCloudQueueShoveledMessage<ConversationMessageWorker>
    {
        [JsonConstructor]
        public ConversationMessageWorker() { }

        
        public ConversationMessageWorker(int regionID) : base(regionID) { }

        public ConversationMessageWorkerType Type { get; set; }

        public DateTime MessageTimestamp { get; set; }
        public string ConversationID { get; set; }
        public string MessageID { get; set; }
        public int UserID { get; set; }
        public string Username { get; set; }
        public DateTime Timestamp { get; set; }
        public ConversationAttachment Attachment { get; set; }
        public string Body { get; set; }
        public bool Unlike { get; set; }
        public ConversationMessage Message { get; set; }
        public int[] Mentions { get; set; }
        public ConversationMessageEmoteSubstitution[] EmoteSubstitutions { get; set; }

        public static void CreateEditAttachment(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username, ConversationAttachment attachment)
        {
            new ConversationMessageWorker()
            {
                Type = ConversationMessageWorkerType.EditAttachment,
                ConversationID = conversationID,
                MessageID = messageID,
                MessageTimestamp = messageTimestamp,
                Attachment = attachment,
                Timestamp = DateTime.UtcNow, 
                UserID = userID,
                Username = username
            }.Enqueue();
        }

        public static void CreateDeleteAttachment(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username)
        {
            new ConversationMessageWorker()
            {
                Type = ConversationMessageWorkerType.DeleteAttachment,
                ConversationID = conversationID,
                MessageID = messageID,
                MessageTimestamp = messageTimestamp,
                Timestamp = DateTime.UtcNow,
                UserID = userID,
                Username = username
            }.Enqueue();
        }

        public static void CreateEditMessage(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username, string body, DateTime editTimestamp, int[] mentions, ConversationMessageEmoteSubstitution[] emoteSubstitutions)
        {
            new ConversationMessageWorker()
            {
                Type = ConversationMessageWorkerType.EditMessage,
                ConversationID = conversationID,
                MessageID = messageID,
                MessageTimestamp = messageTimestamp,
                Timestamp = editTimestamp,
                UserID = userID,
                Username = username,
                Body = body,
                Mentions = mentions,
                EmoteSubstitutions = emoteSubstitutions
            }.Enqueue();
        }

        public static void CreateDeleteMessage(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username, DateTime deleteTimestamp)
        {
            new ConversationMessageWorker()
            {
                Type = ConversationMessageWorkerType.DeleteMessage,
                ConversationID = conversationID,
                MessageID = messageID,
                MessageTimestamp = messageTimestamp,
                Timestamp = deleteTimestamp,
                UserID = userID,
                Username = username
            }.Enqueue();
        }

        public static void CreateLikeMessage(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username, bool unlike)
        {
            new ConversationMessageWorker()
            {
                Type = ConversationMessageWorkerType.LikeMessage,
                ConversationID = conversationID,
                MessageID = messageID,
                MessageTimestamp = messageTimestamp,                
                UserID = userID,
                Username = username,
                Unlike = unlike
            }.Enqueue();
        }

        public static void CreateSetLikeInfo(ConversationMessage message)
        {
            foreach (var region in QueueConfiguration.RemoteRegions)
            {
                new ConversationMessageWorker(region)
                {
                    Type = ConversationMessageWorkerType.UpdateMessageLikes,
                    Message = message
                }.Enqueue();    
            }            
        }

        public static void CreateNewMessage(ConversationMessage message)
        {
            new ConversationMessageWorker()
            {
                Type = ConversationMessageWorkerType.NewMesssage,
                Message = message
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(QueueProcessor_ProcessMessage);
        }

        private static void QueueProcessor_ProcessMessage(ConversationMessageWorker e)
        {
            switch (e.Type)
            {
                case ConversationMessageWorkerType.EditAttachment:
                    ConversationManager.EditAttachment(e.ConversationID, e.MessageID, e.MessageTimestamp, e.UserID, e.Timestamp, e.Username, e.Attachment);
                    break;
                case ConversationMessageWorkerType.DeleteAttachment:
                case ConversationMessageWorkerType.DeleteMessage:
                     ConversationManager.DeleteMessage(e.ConversationID, e.MessageID, e.MessageTimestamp, e.UserID, e.Timestamp, e.Username);
                    break;
                case ConversationMessageWorkerType.EditMessage:
                    ConversationManager.EditMessage(e.ConversationID, e.MessageID, e.MessageTimestamp, e.UserID, e.Timestamp, e.Username, e.Body, e.Mentions ?? new int[0],
                        e.EmoteSubstitutions ?? new ConversationMessageEmoteSubstitution[0]);
                    break;
                case ConversationMessageWorkerType.LikeMessage:
                    if (e.Unlike)
                    {
                        ConversationManager.UnlikeMessage(e.ConversationID, e.MessageID, e.MessageTimestamp, e.UserID, e.Username);
                    }
                    else
                    {
                        ConversationManager.LikeMessage(e.ConversationID, e.MessageID, e.MessageTimestamp, e.UserID, e.Username);
                    }
                    
                    break;
                case ConversationMessageWorkerType.NewMesssage:
                    PersistMessage(e.Message);
                    break;

                case ConversationMessageWorkerType.UpdateMessageLikes:
                    ConversationManager.UpdateMessageLikes(e.Message.ConversationID, e.Message.ID, e.Message.Timestamp.FromEpochMilliconds(), e.Message.LikeCount, e.Message.LikeUserIDs, e.Message.LikeUsernames);
                    break;

            }
        }

        private static void PersistMessage(ConversationMessage message)
        {           
            try
            {
                ConversationManager.Index(message);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to save to Elasticsearch!");
            }                                  
        }
    }
}
