﻿using System;
using System.Collections.Generic;
using Curse.CloudQueue;

namespace Curse.Friends.Data
{
    public enum ExternalCommunityChangeType
    {
        Commissioned,
        Decommissioned,
        Transferred,
        OwnerReauthenticated,
        LinksChanged,
    }

    [CloudQueueProcessor(true)]
    public class ExternalCommunityCoordinator : BaseCloudQueueRoutedMessage<ExternalCommunityCoordinator>
    {
        public ExternalCommunityChangeType ChangeType { get; set; }

        public string ExternalID { get; set; }

        public string[] TransferredExternalIDs { get; set; }

        public ExternalAccount OwnerAccount { get; set; }
        public HashSet<Guid> MappedGroups { get; set; }

        public ExternalCommunityCoordinator()
        {
            
        }

        public ExternalCommunityCoordinator(ExternalCommunity stream)
        {
            stream.EnsureHost();
            ExternalID = stream.ExternalID;
            DestinationRegionID = stream.RegionID;
            DestinationServerName = stream.MachineName;
        }

        public static void StreamCommissioned(ExternalCommunity stream)
        {
            new ExternalCommunityCoordinator(stream)
            {
                ChangeType = ExternalCommunityChangeType.Commissioned
            }.Enqueue();
        }

        public static void StreamDecommissioned(ExternalCommunity stream)
        {
            new ExternalCommunityCoordinator(stream)
            {
                ChangeType = ExternalCommunityChangeType.Decommissioned
            }.Enqueue();
        }

        public static void StreamsTransferred(int regionID, string machineName, string[] externalIDs)
        {
            new ExternalCommunityCoordinator()
            {
                DestinationRegionID = regionID,
                DestinationServerName = machineName,
                ChangeType = ExternalCommunityChangeType.Transferred,
                TransferredExternalIDs = externalIDs
            }.Enqueue();
        }

        public static void OwnerReauthenticated(ExternalCommunity stream, ExternalAccount account)
        {
            new ExternalCommunityCoordinator(stream)
            {
                OwnerAccount =  account,
                ChangeType = ExternalCommunityChangeType.OwnerReauthenticated
            }.Enqueue();
        }

        public static void LinksChanged(ExternalCommunity community)
        {
            new ExternalCommunityCoordinator(community)
            {
                ChangeType = ExternalCommunityChangeType.LinksChanged,
                MappedGroups = community.MappedGroups
            }.Enqueue();
        }
    }
}
