﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Logging;

namespace Curse.Friends.Data
{
    public enum ExternalCommunityMemberIndexChangeType
    {
        AccountLinked,
        AccountUnlinked,
        ExternalMembershipsRemoved,
        ExternalMembershipsAdded,
        Reindex
    }

    [CloudQueueProcessor(true)]
    [CloudQueue(true)]
    public class ExternalCommunityMemberIndexWorker : BaseCloudQueueShoveledMessage<ExternalCommunityMemberIndexWorker>
    {
        private static readonly LogCategory ThrottledLogger = new LogCategory("ExternalCommunityMemberIndexWorker") { Throttle = TimeSpan.FromMinutes(10) };

        public ExternalCommunityMemberIndexChangeType ChangeType { get; set; }

        public AccountType ExternalCommunityType { get; set; }

        public string ExternalCommunityID { get; set; }

        public GroupRoleTag ExternalRoleTag { get; set; }

        public int UserID { get; set; }

        public string[] ExternalUserIDs { get; set; }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        public static void CreateAccountLinked(ExternalAccountMapping mapping)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.AccountLinked,
                ExternalUserIDs = new []{mapping.ExternalID},
                ExternalCommunityType = mapping.Type,
                UserID = mapping.UserID
            }.Enqueue();
        }

        public static void CreateAccountUnlinked(ExternalAccountMapping mapping)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.AccountUnlinked,
                ExternalUserIDs = new []{mapping.ExternalID},
                ExternalCommunityType = mapping.Type,
                UserID = mapping.UserID
            }.Enqueue();
        }

        public static void CreateExternalMembershipsAdded(ExternalCommunity community, GroupRoleTag externalCommunityRole, string[] externalUserIDs)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.ExternalMembershipsAdded,
                ExternalUserIDs = externalUserIDs,
                ExternalCommunityID = community.ExternalID,
                ExternalCommunityType = community.Type,
                ExternalRoleTag = externalCommunityRole
            }.Enqueue();
        }

        public static void CreateExternalMembershipAdded(ExternalCommunityMembership member)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.ExternalMembershipsAdded,
                ExternalUserIDs = new []{member.ExternalUserID},
                ExternalCommunityID = member.ExternalCommunityID,
                ExternalCommunityType = member.Type,
                ExternalRoleTag = member.RoleTag,
            }.Enqueue();
        }

        public static void CreateExternalMembershipRemoved(ExternalCommunityMembership member)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.ExternalMembershipsRemoved,
                ExternalUserIDs = new []{member.ExternalUserID},
                ExternalCommunityID = member.ExternalCommunityID,
                ExternalCommunityType = member.Type,
                ExternalRoleTag = member.RoleTag
            }.Enqueue();
        }

        public static void CreateExternalMembershipsRemoved(ExternalCommunity community, GroupRoleTag externalCommunityRole, string[] externalUserIDs)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.ExternalMembershipsRemoved,
                ExternalUserIDs = externalUserIDs,
                ExternalCommunityID = community.ExternalID,
                ExternalCommunityType = community.Type,
                ExternalRoleTag = externalCommunityRole
            }.Enqueue();
        }

        public static void CreateReindex(ExternalCommunity community)
        {
            return;
            new ExternalCommunityMemberIndexWorker
            {
                ChangeType = ExternalCommunityMemberIndexChangeType.Reindex,                
                ExternalCommunityID = community.ExternalID,
                ExternalCommunityType = community.Type,                
            }.Enqueue();
        }

        private static void Process(ExternalCommunityMemberIndexWorker worker)
        {
            try
            {
                ThrottledLogger.Warn("Skipping index worker for external community members since it is not working properly", worker);
                return;

                switch (worker.ChangeType)
                {
                    case ExternalCommunityMemberIndexChangeType.AccountLinked:
                    {
                        foreach (var externalID in worker.ExternalUserIDs)
                        {
                            foreach (var members in ExternalCommunityMemberManager.Scroll(
                                new ExternalCommunityMemberSearch {ExternalUserID = externalID, ExternalCommunityType = worker.ExternalCommunityType}))
                            {
                                ExternalCommunityMemberManager.BulkUpdateUserID(members, worker.UserID, true);
                            }
                        }
                        break;
                    }
                    case ExternalCommunityMemberIndexChangeType.AccountUnlinked:
                    {
                        foreach (var externalID in worker.ExternalUserIDs)
                        {
                            foreach (var members in ExternalCommunityMemberManager.Scroll(
                                new ExternalCommunityMemberSearch {ExternalUserID = externalID, ExternalCommunityType = worker.ExternalCommunityType}))
                            {
                                ExternalCommunityMemberManager.BulkUpdateUserID(members, worker.UserID, false);
                            }
                        }
                        break;
                    }
                    case ExternalCommunityMemberIndexChangeType.ExternalMembershipsAdded:
                    case ExternalCommunityMemberIndexChangeType.ExternalMembershipsRemoved:
                    {
                        var accounts = ExternalAccount.MultiGetLocal(worker.ExternalUserIDs.Select(id => new KeyInfo(id, worker.ExternalCommunityType)));
                        var linksLookup = ExternalAccountMapping.MultiGetLocal(accounts.SelectMany(a => a.MappedUsers.Select(id => new KeyInfo(id, a.Type, a.ExternalID)))).Where(l => !l.IsDeleted)
                            .ToLookup(l => l.ExternalID);

                        var membersToAdd = new List<ExternalCommunityMemberSearchModel>();

                        foreach (var membership in ExternalCommunityMembership.MultiGetLocal(
                            worker.ExternalUserIDs.Select(id => new KeyInfo(id, worker.ExternalCommunityID, worker.ExternalCommunityType, worker.ExternalRoleTag))))
                        {
                            var member = new ExternalCommunityMemberSearchModel(membership, linksLookup[membership.ExternalUserID].Select(l => l.UserID).ToArray());
                            membersToAdd.Add(member);

                            if (membersToAdd.Count == ExternalCommunityMemberManager.MaxDocumentRequestCount)
                            {
                                ExternalCommunityMemberManager.BulkUpsert(membersToAdd);
                                membersToAdd.Clear();
                            }
                        }
                        if (membersToAdd.Count > 0)
                        {
                            ExternalCommunityMemberManager.BulkUpsert(membersToAdd);
                        }
                        break;
                    }
                    case ExternalCommunityMemberIndexChangeType.Reindex:
                        ReindexExternalCommunityMembers(worker.ExternalCommunityID, worker.ExternalCommunityType);
                        break;
                    default:
                        throw new InvalidOperationException();
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing ExternalCommunityMemberIndexWorker", worker);
            }
        }

        private static void ReindexExternalCommunityMembers(string externalCommunityID, AccountType externalCommunityType)
        {
            // Get the external community
            var externalCommunity = ExternalCommunity.GetLocal(externalCommunityID, externalCommunityType);
            if (externalCommunity == null)
            {
                Logger.Warn("Unable to reindex external community members. The community could not be retreived from the database.", new { externalCommunityID, externalCommunityType });
                return;
            }
           
            // Get all of the members for the community
            var memberships = externalCommunity.GetAllMemberships();

            if (!memberships.Any())
            {
                return;
            }

            var externalAccountKeys = memberships.Select(p => new KeyInfo(p.ExternalUserID, p.Type));
            var externalAccounts = ExternalAccount.MultiGetLocal(externalAccountKeys);

            var linksLookup = ExternalAccountMapping.MultiGetLocal(externalAccounts.SelectMany(a => a.MappedUsers.Select(id => new KeyInfo(id, a.Type, a.ExternalID)))).Where(l => !l.IsDeleted)
                                                    .ToLookup(l => l.ExternalID);

            var membersToAdd = new List<ExternalCommunityMemberSearchModel>();

            foreach (var membership in memberships)
            {
                var member = new ExternalCommunityMemberSearchModel(membership, linksLookup[membership.ExternalUserID].Select(l => l.UserID).ToArray());
                membersToAdd.Add(member);

                if (membersToAdd.Count == ExternalCommunityMemberManager.MaxDocumentRequestCount)
                {
                    ExternalCommunityMemberManager.BulkUpsert(membersToAdd);
                    membersToAdd.Clear();
                }
            }

            if (membersToAdd.Count > 0)
            {
                ExternalCommunityMemberManager.BulkUpsert(membersToAdd);
            }

        }
    }
}
