﻿using System;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    public enum CommunitySyncType
    {
        Linked,
        Unlinked,
        AccountAbandoned
    }

    [CloudQueue(true)]
    [CloudQueueProcessor(true)]
    public class ExternalCommunitySyncWorker : BaseCloudQueueShoveledMessage<ExternalCommunitySyncWorker>
    {
        public string ExternalCommunityID { get; set; }

        public Guid GroupID { get; set; }

        public AccountType CommunityType { get; set; }

        public CommunitySyncType ChangeType { get; set; }

        public ExternalCommunitySyncWorker()
        {
            
        }

        private ExternalCommunitySyncWorker(int regionID) : base(regionID)
        {
            
        }

        public static void Linked(int regionID, string externalID, AccountType type, Guid groupID)
        {
            new ExternalCommunitySyncWorker(regionID)
            {
                ChangeType =  CommunitySyncType.Linked,
                ExternalCommunityID = externalID,
                CommunityType = type,
                GroupID = groupID
            }.Enqueue();
        }

        public static void Unlinked(int regionID, string externalID, AccountType type, Guid groupID)
        {
            new ExternalCommunitySyncWorker(regionID)
            {
                ChangeType = CommunitySyncType.Unlinked,
                ExternalCommunityID = externalID,
                CommunityType = type,
                GroupID = groupID
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(ExternalCommunitySyncWorker worker)
        {
            try
            {
                var community = ExternalCommunity.GetLocal(worker.ExternalCommunityID, worker.CommunityType);
                if (community == null)
                {
                    return;
                }

                var mapping = ExternalCommunityMapping.GetLocal(worker.GroupID, worker.ExternalCommunityID, worker.CommunityType);
                if (mapping == null)
                {
                    return;
                }

                var group = Group.GetLocal(mapping.GroupID);
                if (group == null)
                {
                    return;
                }

                var members = ExternalCommunityMembership.GetAllLocal(m => m.CommunityAndTypeIndex, worker.ExternalCommunityID + worker.CommunityType)
                    .Where(m => m.Status != ExternalCommunityMembershipStatus.Deleted).ToArray();

                switch (worker.ChangeType)
                {
                    case CommunitySyncType.Linked:
                        Logger.Info(string.Format("Linking external community {0} to group {1}", community.ExternalName, group.Title), new { mapping, group });
                        foreach (var user in members)
                        {
                            mapping.AddMemberRoleMapping(user);
                        }
                        ExternalCommunityMemberSyncWorker.Create(community.RegionID, community.ExternalID, community.Type);
                        
                        ExternalCommunityLinkChangedCoordinator.Create(group, ExternalCommunityLinkChangeType.Linked, community);
                        break;

                    case CommunitySyncType.Unlinked:
                        Logger.Info(string.Format("Unlinking external community {0} from group {1}",community.ExternalName, group.Title), new {mapping, group});
                        foreach (var user in members)
                        {
                            mapping.RemoveMemberRoleMapping(user);
                        }
                        ExternalCommunityMemberSyncWorker.Create(community.RegionID, community.ExternalID, community.Type);

                        ExternalCommunityLinkChangedCoordinator.Create(group, ExternalCommunityLinkChangeType.Unlinked, community);
                        break;

                    default:
                        Logger.Warn("Unknown CommunitySyncType", worker);
                        return;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process worker.", worker);
            }
        }

    }
}
