﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [CloudQueue(true)]
    [CloudQueueProcessor(true)]
    public class ExternalGuildMemberSyncWorker : BaseCloudQueueWorkerMessage<ExternalGuildMemberSyncWorker>
    {
        public AccountType AccountType { get; set; }

        public ExternalGuildIdentifier GuildIdentifier { get; set; }

        public static void Create(AccountType accountType, ExternalGuildIdentifier guildIdentifier)
        {
            new ExternalGuildMemberSyncWorker
            {
                AccountType = accountType,
                GuildIdentifier = guildIdentifier
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        public static void Process(ExternalGuildMemberSyncWorker worker)
        {
            try
            {
                var guild = ExternalGuild.Get(worker.GuildIdentifier);
                if (guild == null)
                {
                    Logger.Warn("Couldn't find external guild", worker);
                    return;
                }

                Logger.Debug("Processing guild memberships", new {worker, guild});

                var guildIndex = guild.GetGuildIndex();

                // Front-load all DB calls
                var memberships = ExternalGuildMember.GetAllLocal(m => m.GuildIndex, guildIndex);

                var membershipsByRole = memberships.ToLookup(m => m.GuildRole);

                var accounts = ExternalAccount.MultiGetLocal(memberships.Where(m => !string.IsNullOrEmpty(m.AccountID) && m.Type == guild.Type)
                    .GroupBy(g => Tuple.Create(g.AccountID, worker.AccountType)).Select(t => new KeyInfo(t.Key.Item1, t.Key.Item2)));
                var links = ExternalAccountMapping.MultiGetLocal(accounts.SelectMany(g => g.MappedUsers.Select(u => new KeyInfo(u, g.Type, g.ExternalID))));

                var linksByExternalUser = links.ToLookup(g => g.ExternalID);
                var linksByUser = links.ToLookup(g => g.UserID);
                var uniqueUsers = new HashSet<int>(accounts.SelectMany(m => m.MappedUsers));

                foreach (var groupID in guild.MappedGroups)
                {
                    var group = Group.GetWritableByID(groupID);
                    if (group == null)
                    {
                        continue;
                    }

                    var allGroupRoles = group.GetRoles(true);
                    var groupRoles = allGroupRoles.Where(r => r.Source == worker.AccountType && r.SyncID == guildIndex).GroupBy(r => r.Tag).ToDictionary(r => r.Key, r => r.First());

                    var members = group.GetMembers(uniqueUsers).ToDictionary(m => m.UserID);
                    foreach (var membership in memberships)
                    {
                        var membersWithMapping = new List<GroupMember>();
                        foreach (var link in linksByExternalUser[membership.AccountID])
                        {
                            GroupMember member;
                            if (members.TryGetValue(link.UserID, out member))
                            {
                                membersWithMapping.Add(member);
                            }
                        }

                        if (membersWithMapping.Count == 0)
                        {
                            continue;
                        }

                        GroupRole groupRole;
                        if (groupRoles.TryGetValue(membership.GuildRole, out groupRole))
                        {
                            foreach (var member in membersWithMapping)

                                if (linksByUser[member.UserID].Any(l => !l.IsDeleted && membershipsByRole[groupRole.Tag].Any(m => m.AccountID == l.ExternalID)))
                                {
                                    if (!member.Roles.Contains(groupRole.RoleID) && !groupRole.IsDeleted)
                                    {
                                        group.SystemAddMemberRole(groupRole, member, allGroupRoles);
                                    }
                                }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error occurred while processing community memberships!", worker);
            }
        }
    }
}
