﻿using System;
using System.Linq;
using Curse.CloudQueue;
using Curse.Friends.Data.DerivedModels;

namespace Curse.Friends.Data
{
    public enum ExternalGuildSyncType
    {
        Linked,
        Unlinked,
    }

    [CloudQueue(true)]
    [CloudQueueProcessor(true)]
    public class ExternalGuildSyncWorker : BaseCloudQueueWorkerMessage<ExternalGuildSyncWorker>
    {
        public ExternalGuildIdentifier GuildIdentifier { get; set; }

        public Guid GroupID { get; set; }

        public CommunitySyncType ChangeType { get; set; }

        public static void Linked(ExternalGuildIdentifier guildIdentifier, Guid groupID)
        {
            new ExternalGuildSyncWorker
            {
                ChangeType =  CommunitySyncType.Linked,
                GroupID = groupID,
                GuildIdentifier = guildIdentifier
            }.Enqueue();
        }

        public static void Unlinked(ExternalGuildIdentifier guildIdentifier, Guid groupID)
        {
            new ExternalGuildSyncWorker
            {
                ChangeType = CommunitySyncType.Unlinked,
                GroupID = groupID,
                GuildIdentifier = guildIdentifier
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(ExternalGuildSyncWorker worker)
        {
            try
            {
                var guild = ExternalGuild.Get(worker.GuildIdentifier);
                if (guild == null)
                {
                    return;
                }

                var mapping = guild.GetMapping(worker.GroupID);
                if (mapping == null)
                {
                    return;
                }

                var group = Group.GetLocal(mapping.GroupID);
                if (group == null)
                {
                    return;
                }

                var guildIndex = guild.GetGuildIndex();

                switch (worker.ChangeType)
                {
                    case CommunitySyncType.Linked:
                        Logger.Info(string.Format("Linking external guild {0} to group {1}", guildIndex, group.Title), new {mapping, group});
                        ExternalGuildMemberSyncWorker.Create(guild.Type, worker.GuildIdentifier);
                        break;

                    case CommunitySyncType.Unlinked:
                        Logger.Info(string.Format("Unlinking external community {0} from group {1}", guildIndex, group.Title), new { mapping, group });
                        ExternalGuildMemberSyncWorker.Create(guild.Type, worker.GuildIdentifier);
                        break;

                    default:
                        Logger.Warn("Unknown CommunitySyncType", worker);
                        return;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process worker.", worker);
            }
        }
    }
}
