﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Extensions;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [CloudQueue(true)]
    [CloudQueueProcessor(true)]
    public class ExternalGuildUserSyncWorker : BaseCloudQueueWorkerMessage<ExternalGuildUserSyncWorker>
    {
        public int UserID { get; set; }

        public AccountType AccountType { get; set; }

        public static void Create(int userID, AccountType accountType)
        {
            new ExternalGuildUserSyncWorker()
            {
                UserID = userID,
                AccountType = accountType
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(ExternalGuildUserSyncWorker worker)
        {
            try
            {
                var links = ExternalAccountMapping.GetAllLocal(l => l.UserID, worker.UserID)
                    .Where(l => l.Type == worker.AccountType)
                    .GroupBy(a => a.ExternalID)
                    .ToDictionary(a => a.Key, a => a.First());

                // Mark account mappings as having a full sync performed
                foreach (var link in links.Values)
                {
                    link.LastMembershipSyncTime = DateTime.UtcNow;
                    link.Update(l => l.LastMembershipSyncTime);
                }

                // Get Guild Memberships, groups, group memberships, and group roles
                var allMemberships = links.Values.SelectMany(l => ExternalGuildMember.GetAllLocal(m => m.AccountID, l.ExternalID)).GroupBy(m => m.GetGuildInfo()).ToArray();
                var allGuilds = ExternalGuild.MultiGetLocal(allMemberships.Select(s => new KeyInfo(s.Key.Type, s.Key.GameRegion, s.Key.GameServer, s.Key.Name))).ToDictionary(m => m.GetGuildInfo());
                var allGroups = Group.MultiGetLocal(new HashSet<Guid>(allGuilds.SelectMany(g => g.Value.MappedGroups)).Select(id => new KeyInfo(id))).ToDictionary(g => g.GroupID);
                var allGroupRoles = allGroups.SelectMany(g => g.Value.GetRoles(true)).GroupBy(g => g.GroupID).ToDictionary(g => g.Key, g => g.ToArray());
                var allGroupMemberships = new Dictionary<Guid, GroupMember>();
                foreach (var region in allGroups.GroupBy(g => g.Value.RegionID))
                {
                    var groupMembers = GroupMember.MultiGet(region.Key, region.Select(r => new KeyInfo(r.Key, worker.UserID)));
                    foreach (var groupMember in groupMembers)
                    {
                        allGroupMemberships[groupMember.GroupID] = groupMember;
                    }
                }

                foreach (var membershipGrouping in allMemberships)
                {
                    ExternalGuild guild;
                    if (!allGuilds.TryGetValue(membershipGrouping.Key, out guild))
                    {
                        // No guild to sync
                        continue;
                    }

                    if (guild.MappedGroups.Count == 0)
                    {
                        // No groups
                        continue;
                    }

                    var guildIndex = guild.GetGuildIndex();
                    var roleGroupings = membershipGrouping.Where(m=>m.GuildIndex==guildIndex).GroupBy(m => m.GuildRole).ToArray();
                    foreach (var groupID in guild.MappedGroups)
                    {
                        Group group;
                        if (!allGroups.TryGetValue(groupID, out group))
                        {
                            continue;
                        }

                        GroupMember member;
                        if (!allGroupMemberships.TryGetValue(groupID, out member))
                        {
                            continue;
                        }

                        GroupRole[] groupRoles;
                        if (!allGroupRoles.TryGetValue(groupID, out groupRoles))
                        {
                            continue;
                        }

                        foreach (var roleGrouping in roleGroupings)
                        {
                            var role = groupRoles.FirstOrDefault(r => r.SyncID == guildIndex && r.Source == guild.Type && r.Tag == roleGrouping.Key);
                            if (role == null)
                            {
                                continue;
                            }

                            var shouldHaveRole = roleGrouping.Select(m => links.GetValueOrDefault(m.AccountID)).Any(m => m != null && !m.IsDeleted);
                            if (shouldHaveRole && !member.Roles.Contains(role.RoleID) && !role.IsDeleted)
                            {
                                // Add the role to the member
                                group.SystemAddMemberRole(role, member, groupRoles);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process worker.", worker);
            }
        }

    }
}
