﻿using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Friends.Data.Search;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Curse.Friends.Data.Search.FriendSearch;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [CloudQueueProcessor(8, true)]
    public class FriendHintSearchIndexer : BaseCloudQueueShoveledMessage<FriendHintSearchIndexer>
    {        
        private static readonly HashSet<int> PersonsOfInterest = new HashSet<int>(new[] { 1, 817369, 7231064, 14923652, 59128, 65, 438695, 16203539, 2547696, 2794552, 4257619, 4, 128, 2566586, 11727540, 16470923, 16018173 });

        public static void CreateForUser(int userID)
        {
            new FriendHintSearchIndexer { UserID = userID }.Enqueue();
        }

        public static void CreateForHint(FriendHint friendHint)
        {
            new FriendHintSearchIndexer { FriendHint = friendHint }.Enqueue();
        }

        public FriendHint FriendHint
        {
            get; set;
        }

        public int? UserID
        {
            get;
            set;
        }

        public bool RebuildIndex
        {
            get;
            set;
        }

        public static void StartProcessor()
        {
            StartProcessor(QueueProcessor_ProcessMessage);
        }

        static void QueueProcessor_ProcessMessage(FriendHintSearchIndexer e)
        {
            Reindex(e.FriendHint, e.UserID, e.RebuildIndex);
        }

        public static void Reindex(FriendHint friendHint = null, int? userID = null, bool rebuildIndex = false)
        {
            if (friendHint != null)
            {
                ProcessHints(new[] { friendHint });
            }
            else if (userID.HasValue)
            {
                var hints = FriendHint.GetAllLocal(p => p.UserID, userID.Value);
                if (!hints.Any())
                {
                    var userRegion = UserRegion.GetByUserID(userID.Value, true);
                    if (userRegion == null)
                    {
                        Logger.Warn("Unable to process hints. The list is empty, and user region is unknown.", new { userID });
                        return;
                    }

                    if (userRegion.RegionID != QueueConfiguration.LocalRegionID)
                    {                        
                        hints = FriendHint.GetAll(userRegion.RegionID, p => p.UserID, userID.Value);
                    }

                    if (!hints.Any())
                    {
                        Logger.Debug("Unable to process hints. The list is empty.", new { userID });
                        return;
                    }
                }

                ProcessHints(hints);
            }
            else
            {
                if (rebuildIndex)
                {
                    CharacterFriendSearchManager.DeleteIndex();
                    PlatformFriendSearchManager.DeleteIndex();
                    UsernameFriendSearchManager.DeleteIndex();
                    EmailFriendSearchManager.DeleteIndex();
                }

                var sw = Stopwatch.StartNew();
                Logger.Info("Reindexing all friend searches.");
                FriendHint.BatchOperateOnIndexLocal(p => p.IndexMode, IndexMode.Default, 1000, hints => ProcessHints(hints.ToArray(), true));
                Logger.Info("Finished reindexing all friend searches in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds");
            }
        }

        static void ProcessHints(FriendHint[] hints, bool verboseLogging = false)
        {
            hints = hints.Where(p => !string.IsNullOrEmpty(p.SearchTerm)).ToArray();

            if (!hints.Any())
            {
                Logger.Debug("Unable to process hints. The list is empty.");
                return;
            }

            var firstUserID = hints.FirstOrDefault().UserID;

            if (PersonsOfInterest.Contains(firstUserID))
            {
                Logger.Trace("Performing friend hint search index for person of interest: " + firstUserID, new { HintCount = hints.Length });
            }

            var usernameSearches = new List<UsernameFriendSearchModel>();
            var emailSearches = new List<EmailFriendSearchModel>();
            var characterSearches = new List<CharacterFriendSearchModel>();
            var characterDeletions = new List<string>();
            var platformSearches = new List<PlatformFriendSearchModel>();
            var platformDeletions = new List<string>();

            if (verboseLogging)
            {
                Logger.Debug("Indexing " + hints.Length + " more friend hints...");
            }

            foreach (var hint in hints)
            {               
                var userID = hint.UserID;
                var userStatistics = UserStatistics.GetByUserOrDefault(userID);

                if (PersonsOfInterest.Contains(firstUserID))
                {
                    Logger.Trace("Process hint for user: " + firstUserID, hint);
                }
                
                switch (hint.Type)
                {
                    case FriendHintType.Username:
                        {
                            var searchModel = new UsernameFriendSearchModel
                            {
                                FriendCount = userStatistics.FriendCount,
                                UserID = userID,
                                Username = hint.SearchTerm,
                                AvatarUrl = hint.AvatarUrl,
                                IsMerged = hint.IsMerged
                            };
                            usernameSearches.Add(searchModel);

                            break;
                        }
                    case FriendHintType.Email:
                        {
                            var searchModel = new EmailFriendSearchModel
                            {
                                FriendCount = userStatistics.FriendCount,
                                UserID = userID,
                                EmailAddress = hint.SearchTerm.ToLowerInvariant(),
                                AvatarUrl = hint.AvatarUrl,
                                IsMerged = hint.IsMerged
                            };
                            emailSearches.Add(searchModel);
                            break;
                        }

                    case FriendHintType.Game:
                        {
                            var searchModel = new CharacterFriendSearchModel(userID, hint.SearchTerm, hint.GameID, hint.Server, hint.Region)
                            {
                                FriendCount = userStatistics.FriendCount,
                                CharacterName = hint.DisplayName??hint.SearchTerm,
                                AvatarUrl = hint.AvatarUrl,
                                Description = hint.Description,
                            };
                            if (hint.Status == FriendHintStatus.Deleted)
                            {
                                characterDeletions.Add(searchModel.UniqueKey);
                            }
                            else
                            {
                                characterSearches.Add(searchModel);
                            }
                            break;
                        }
                    case FriendHintType.Platform:
                        {
                            var searchModel = new PlatformFriendSearchModel(userID, hint.SearchTerm, hint.Platform)
                            {
                                FriendCount = userStatistics.FriendCount,
                                AvatarUrl = hint.AvatarUrl,
                                Identity = hint.GetPublicName(),
                                Visibility = hint.Visibility,
                                Description = hint.Description,
                            };

                            if (hint.Status == FriendHintStatus.Deleted)
                            {
                                platformDeletions.Add(searchModel.UniqueKey);
                            }
                            else
                            {
                                platformSearches.Add(searchModel);
                            }
                            break;
                        }
                }

                UsernameFriendSearchManager.IndexMany(usernameSearches, 1000);
                EmailFriendSearchManager.IndexMany(emailSearches, 1000);
                CharacterFriendSearchManager.IndexMany(characterSearches, 1000);
                CharacterFriendSearchManager.DeleteMany(characterDeletions, 1000);
                PlatformFriendSearchManager.IndexMany(platformSearches, 1000);
                PlatformFriendSearchManager.DeleteMany(platformDeletions, 1000);
            }

            UsernameFriendSearchManager.IndexMany(usernameSearches);
            EmailFriendSearchManager.IndexMany(emailSearches);
            CharacterFriendSearchManager.IndexMany(characterSearches);
            CharacterFriendSearchManager.DeleteMany(characterDeletions);
            PlatformFriendSearchManager.IndexMany(platformSearches);
            PlatformFriendSearchManager.DeleteMany(platformDeletions);
        }

    }
}


