﻿using Curse.CloudQueue;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.Statistics;
using Curse.Logging;
using Nest;
using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using System.Diagnostics;
using Curse.CloudSearch;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Search.FriendSearch;

namespace Curse.Friends.Data
{
    [CloudQueueProcessor(4, true)]
    public class FriendListSearchWorker : BaseCloudQueueWorkerMessage<FriendListSearchWorker>
    {
        private static readonly HashSet<int> PersonsOfInterest = new HashSet<int>(new[] { 1, 817369, 7231064, 14923652, 59128, 65, 438695, 16203539, 2547696, 2794552, 4257619, 4, 128, 2566586, 11727540, 16470923, 16018173 });

        /// <summary>
        /// The ID of the receiving user
        /// </summary>
        public int UserID
        {
            get;
            set;
        }

        /// <summary>
        /// The idnetity to use for any suggestions created
        /// </summary>
        public FriendHint Identity
        {
            get;
            set;
        }



        /// <summary>
        /// The friends list of the user in question
        /// </summary>
        public FriendHint[] FriendsList
        {
            get;
            set;
        }


        public static void StartProcessor()
        {
            StartProcessor(QueueProcessor_ProcessMessage);
        }

        private class FriendListSearchResult
        {
            public FriendListSearchResult(FriendHint knownIdentity)
            {
                KnownIdentity = knownIdentity;
            }

            public IUserSearchModel Document
            {
                get;
                set;
            }

            public FriendHint KnownIdentity
            {
                get;
                private set;
            }

            public bool IsValid
            {
                get
                {
                    return Document != null;
                }
            }
        }

        static FriendListSearchResult[] GetCharacterResults(FriendHint[] hints)
        {
            // Iterate over each friend hint in the list, and look for a matching hint in our search provider
            var characterClient = CharacterFriendSearchManager.GetClient();
            var req = new MultiSearchRequest();
            req.Operations = new Dictionary<string, ISearchRequest>();

            var results = new Dictionary<string, FriendListSearchResult>();

            foreach (var hint in hints)
            {
                QueryContainer queryContainer = null;
                var mustFilters = new List<QueryContainer>();
                var boolQuery = new BoolQuery();

                // Game ID              
                mustFilters.Add(Query<CharacterFriendSearchModel>.Term(p => p.GameID, hint.GameID));

                var sanitizedName = CharacterFriendSearchModel.SanitizeSearchTerm(hint.SearchTerm);

                // Character name
                mustFilters.Add(Query<CharacterFriendSearchModel>.Term(p => p.SearchTerm, sanitizedName));

                // Region
                if (!string.IsNullOrEmpty(hint.Region))
                {
                    mustFilters.Add(Query<CharacterFriendSearchModel>.Term(p => p.ServerRegion, hint.Region));
                }

                // Server
                if (!string.IsNullOrEmpty(hint.Server))
                {
                    mustFilters.Add(Query<CharacterFriendSearchModel>.Term(p => p.ServerName, hint.Server));
                }

                boolQuery.Must = mustFilters;
                queryContainer &= boolQuery;

                var key = "Game:" + hint.GameID + ";SearchTerm:" + hint.SearchTerm + ";Region:" + hint.Region.ToEmptyWhenNull() + ";Server:" + hint.Server.ToEmptyWhenNull();

                if (req.Operations.ContainsKey(key))
                {
                    Logger.Debug("Duplicate character hint found. It will not be used for searching!", new { Key = key });
                    continue;
                }

                results.Add(key, new FriendListSearchResult(hint));
                var searchRequest = new SearchRequest<CharacterFriendSearchModel>();
                searchRequest.Query = queryContainer;
                req.Operations.Add(key, searchRequest);
            }

            var search = characterClient.MultiSearch(req);

            foreach (var result in results)
            {
                var resp = search.GetResponse<CharacterFriendSearchModel>(result.Key);
                if (resp != null)
                {
                    result.Value.Document = resp.Documents.OrderByDescending(p => p.FriendCount).FirstOrDefault();
                }
            }

            return results.Values.Where(p => p.IsValid).ToArray();

        }

        static FriendListSearchResult[] GetPlatformResults(FriendHint[] hints)
        {
            // Iterate over each friend hint in the list, and look for a matching hint in our search provider
            var characterClient = PlatformFriendSearchManager.GetClient();
            var req = new MultiSearchRequest();
            req.Operations = new Dictionary<string, ISearchRequest>();

            var results = new Dictionary<string, FriendListSearchResult>();

            foreach (var hint in hints)
            {

                if (string.IsNullOrWhiteSpace(hint.SearchTerm))
                {
                    continue;
                }

                QueryContainer queryContainer = null;
                var mustFilters = new List<QueryContainer>();
                var boolQuery = new BoolQuery();

                // Game ID              
                mustFilters.Add(Query<PlatformFriendSearchModel>.Term(p => p.Platform, hint.Platform));

                // Character name
                mustFilters.Add(Query<PlatformFriendSearchModel>.Term(p => p.SearchTerm.Suffix(CloudSearchConstants.LowercaseSuffix), hint.SearchTerm.ToLowerInvariant()));

                boolQuery.Must = mustFilters;
                queryContainer &= boolQuery;

                var key = "Platform:" + hint.Platform.ToString() + ";SearchTerm:" + hint.SearchTerm;
                if (req.Operations.ContainsKey(key))
                {
                    continue;
                }

                results.Add(key, new FriendListSearchResult(hint));
                var searchRequest = new SearchRequest<PlatformFriendSearchModel>();
                searchRequest.Query = queryContainer;
                req.Operations.Add(key, searchRequest);
            }

            var search = characterClient.MultiSearch(req);

            foreach (var result in results)
            {
                var resp = search.GetResponse<PlatformFriendSearchModel>(result.Key);
                if (resp != null)
                {
                    result.Value.Document = resp.Documents.OrderByDescending(p => p.FriendCount).FirstOrDefault();
                }
            }

            return results.Values.Where(p => p.IsValid).ToArray();
        }

        static FriendSuggestion CreateSuggestion(UserRegion myRegion, FriendSuggestion newSuggestion)
        {            
            
            // See if the user has a suggestion for any of these characters:
            var existingSuggestion = FriendSuggestion.Get(myRegion.RegionID, myRegion.UserID, newSuggestion.OtherUserID);

            if (existingSuggestion != null)
            {
                // If they are equivalent, just update
                if (newSuggestion.IsEquivalent(existingSuggestion))
                {
                    if (existingSuggestion.Status == FriendSuggestionStatus.Pending)
                    {
                        existingSuggestion.FriendCount = newSuggestion.FriendCount;
                        existingSuggestion.AvatarUrl = newSuggestion.AvatarUrl;
                        existingSuggestion.Update();
                    }

                    return null;
                }

                // If the existing suggestion type is better or equal to the existing one, and it is pending, leave it alone.
                if (newSuggestion.Type <= existingSuggestion.Type)
                {
                    return null;
                }

                // If the user has already declined an equivalent or higher suggestion, do not bother them
                if (existingSuggestion.Type >= newSuggestion.Type && existingSuggestion.Status == FriendSuggestionStatus.Declined)
                {
                    return null;
                }

                Logger.Trace("Found an existing suggestion, but we have decided to upgrade it.", new { newSuggestion, existingSuggestion });
            }


            newSuggestion.Insert(myRegion.RegionID);
            
            if (newSuggestion.Type == FriendSuggestionType.GameFriend)
            {
                FriendsStatsManager.Current.GameSuggestions.Track();
            }
            else if (newSuggestion.Type == FriendSuggestionType.PlatformFriend)
            {
                FriendsStatsManager.Current.PlatformSuggestions.Track();
            }

            return newSuggestion;
            
        }

        static string GetSuggestionAvatarUrl(FriendHintType type, FriendPlatform platform, int gameID, string defaultAvatar)
        {
            switch (type)
            {
                case FriendHintType.Platform:
                    return "http://clientupdate-v6.cursecdn.com/PlatformAssets/" + (int)platform + "/Icon128.png";                    
                case FriendHintType.Game:
                    return "http://clientupdate-v6.cursecdn.com/GameAssets/" + gameID + "/Icon128.png";                                    
            }

            return defaultAvatar;
        }

        static string GetFriendlyBattleTag(string battleTag)
        {
            if(battleTag.Contains("#"))
            {
                return battleTag.Substring(0, battleTag.IndexOf("#"));
            }

            return battleTag;
            
        }

        static string GetMyKnownIdentity(int userID, FriendHintType type, FriendPlatform platform, string mySearchTerm, string myRealName, string otherSearchTerm, string knownIdentity)
        {
            if(type != FriendHintType.Platform)
            {
                return myRealName;
            }

            switch(platform)
            {
                case FriendPlatform.BattleNet:
                    
                    Logger.Trace("Getting known identity for BattleNet.", new { mySearchTerm, myRealName, otherSearchTerm, knownIdentity });

                    var theirBattleTag = GetFriendlyBattleTag(otherSearchTerm);
                    if (theirBattleTag.Equals(knownIdentity, StringComparison.InvariantCultureIgnoreCase))
                    {
                        Logger.Trace("Using Battletag: " + mySearchTerm);                        
                        return GetFriendlyBattleTag(mySearchTerm);
                    }
                    else
                    {
                        Logger.Trace("Using real name:" + myRealName);
                        return myRealName;
                    }

                default:
                    return myRealName;
            }


        }

        static void QueueProcessor_ProcessMessage(FriendListSearchWorker e)
        {

            if(e.UserID <= 0 || e.Identity == null || e.FriendsList.Length == 0)
            {
                return;
            }

            var sw = Stopwatch.StartNew();
            
            try
            {
                if (PersonsOfInterest.Contains(e.UserID))
                {
                    Logger.Trace("Performing friend list search for a person of interest: " + e.Identity.DisplayName + "(" + e.Identity.SearchTerm + ") ", 
                        new 
                        { 
                            e.Identity,
                            e.UserID,                            
                            e.FriendsList 
                        });
                }
                
                var myRegion = UserRegion.GetLocal(e.UserID);
                if (myRegion == null)
                {
                    return;
                }

                var me = myRegion.GetUser();
                if (me == null)
                {
                    return;
                }

                // Get the user's friend ID list, regardless of status!
                var friendIDs = new HashSet<int>(Friendship.GetAll(myRegion.RegionID, p => p.UserID, e.UserID).Select(p => p.OtherUserID));
                friendIDs.Add(e.UserID);

                var newSuggestions = new List<FriendSuggestion>();
                var results = new List<FriendListSearchResult>();

                var groups = e.FriendsList.GroupBy(p => p.Type);

                foreach (var group in groups)
                {
                    switch (group.Key)
                    {
                        case FriendHintType.Game:
                            results.AddRange(GetCharacterResults(e.FriendsList));
                            break;

                        case FriendHintType.Platform:
                            results.AddRange(GetPlatformResults(e.FriendsList));
                            break;
                    }
                }

                if (PersonsOfInterest.Contains(e.UserID))
                {
                    Logger.Trace("Found " + results.Count + " results!", results);
                }

                if (!results.Any())
                {
                    return;
                }

                var uniqueUserIDs = new HashSet<int>(results.Select(r => r.Document.UserID)) {e.UserID};
                var userStatistics = UserStatistics.GetDictionaryByUserIDs(uniqueUserIDs);
                var userPrivacy = UserPrivacySettings.GetDictionaryByUserIDs(uniqueUserIDs);

                var myStatistics = userStatistics.GetValueOrDefault(e.UserID);
                var myPrivacy = userPrivacy.GetValueOrDefault(e.UserID);
                foreach (var result in results)
                {
                    // How the requesting user knows this other user
                    var knownIdentity = result.KnownIdentity;

                    // 
                    var doc = result.Document;

                    if (friendIDs.Contains(doc.UserID))
                    {
                        continue;
                    }

                    var newSuggestion = new FriendSuggestion
                    {
                        UserID = e.UserID,
                        OtherUserID = doc.UserID,
                        AvatarUrl = doc.AvatarUrl,
                        FriendCount = doc.FriendCount,
                        Status = FriendSuggestionStatus.Pending,
                        DateSuggested = DateTime.UtcNow
                    };

                    // What identity I am happy to reveal to this user
                    var myKnownIdentity = GetMyKnownIdentity(e.UserID, knownIdentity.Type, knownIdentity.Platform, e.Identity.SearchTerm, e.Identity.DisplayName, knownIdentity.SearchTerm, knownIdentity.DisplayName);

                    // How I know the other user
                    var otherKnownIdentity = knownIdentity.DisplayName;

                    // Construct a new suggestion class, then merge or add it below:
                    if (doc is CharacterFriendSearchModel)
                    {
                        var charModel = (CharacterFriendSearchModel)doc;
                        newSuggestion.Type = FriendSuggestionType.GameFriend;
                        newSuggestion.GameID = charModel.GameID;
                        newSuggestion.Username = otherKnownIdentity;

                    }
                    else if (doc is PlatformFriendSearchModel)
                    {
                        var platformModel = (PlatformFriendSearchModel)doc;
                        newSuggestion.Type = FriendSuggestionType.PlatformFriend;
                        newSuggestion.Platform = platformModel.Platform;
                        newSuggestion.Username = otherKnownIdentity;
                    }
                    else
                    {
                        Logger.Warn("Unknown document type: " + doc.GetType().FullName);
                        continue;
                    }
                    
                    newSuggestion.RequestUsername = myKnownIdentity;
                    newSuggestion.RequestAvatarUrl = GetSuggestionAvatarUrl(knownIdentity.Type, knownIdentity.Platform, knownIdentity.GameID, null);
                    newSuggestion.FriendCount = doc.FriendCount;
                    newSuggestion.AvatarUrl = GetSuggestionAvatarUrl(knownIdentity.Type, knownIdentity.Platform, knownIdentity.GameID, doc.AvatarUrl);
                    var mySuggestion = CreateSuggestion(myRegion, newSuggestion);

                    if (mySuggestion != null)
                    {
                        newSuggestions.Add(mySuggestion);

                        var otherUserRegion = UserRegion.GetLocal(newSuggestion.OtherUserID);
                        if (otherUserRegion == null)
                        {
                            Logger.Error("Unable to create other friend suggestion. Other user region info is missing!", newSuggestion.OtherUserID);
                            continue;
                        }

                        // Determine if we should create the inverse
                        var otherSuggestion = FriendSuggestion.Get(otherUserRegion.RegionID, otherUserRegion.UserID, myRegion.UserID);
                        if (otherSuggestion != null)
                        {
                            continue;
                        }

                        // No suggestion exists, so let's make this suggestion!
                        otherSuggestion = new FriendSuggestion
                        {
                            UserID = otherUserRegion.UserID,
                            OtherUserID = me.UserID,
                            AvatarUrl = newSuggestion.RequestAvatarUrl,
                            Username = newSuggestion.RequestUsername,
                            GameID = newSuggestion.GameID,
                            Platform = newSuggestion.Platform,
                            FriendCount = me.FriendCount,
                            RequestAvatarUrl = newSuggestion.AvatarUrl,
                            RequestUsername = newSuggestion.Username,
                            Status = FriendSuggestionStatus.Pending,
                            Type = newSuggestion.Type,
                            DateSuggested = DateTime.UtcNow
                        };

                        otherSuggestion.Insert(otherUserRegion.RegionID);

                        if (otherSuggestion.IsValidSuggestion(otherUserRegion.UserID, myStatistics, myPrivacy))
                        {
                            // Send a notificiation to each of this user's connected endpoints
                            foreach (var endpoint in ClientEndpoint.GetAllConnected(otherUserRegion.UserID))
                            {
                                FriendSuggestionNotifier.Create(endpoint, new[] { otherSuggestion });
                            }
                        }
                    }
                }
                if (PersonsOfInterest.Contains(e.UserID))
                {
                    Logger.Trace("Found " + newSuggestions.Count + " new suggestions!", new {e.UserID, NewSuggestions = newSuggestions});
                }

                if (!newSuggestions.Any())
                {
                    return;
                }                                

                // Take the top 20 friends, by the friend count
                var suggestedFriends = FriendSuggestion.GetValidFriendSuggestions(e.UserID, newSuggestions.ToArray(), userStatistics, userPrivacy)
                    .OrderByDescending(p => p.FriendCount).Take(100).ToArray();

                if (PersonsOfInterest.Contains(e.UserID))
                {
                    Logger.Trace("Sending " + suggestedFriends.Length + " suggestions to the user!",
                        new {e.UserID, Suggestions = suggestedFriends});
                }

                // Send a notificiation to each of this user's connected endpoints
                foreach (var endpoint in ClientEndpoint.GetAllConnected(e.UserID))
                {
                    FriendSuggestionNotifier.Create(endpoint, suggestedFriends);
                }
            }                
            finally
            {
                sw.Stop();
                if (PersonsOfInterest.Contains(e.UserID))
                {
                    Logger.Trace(
                        "Completed friend list search job in " + sw.Elapsed.TotalMilliseconds.ToString("###,##0.00") +
                        " milliseconds", new {e.UserID, ListCount = e.FriendsList.Length});
                }
            }            
        }

    }
}
