﻿using System;
using System.Web;
using Curse.CloudQueue;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [CloudQueueProcessor(true)]
    public class FriendshipChangeResolver : BaseCloudQueueWorkerMessage<FriendshipChangeResolver>
    {
        /// <summary>
        /// The ID of the user who initiated this change
        /// </summary>
        public int InitiatingUserID
        {
            get;
            set;
        }

        /// <summary>
        /// The ID of the receiving user
        /// </summary>
        public int UserID
        {
            get;
            set;
        }
        
        public Friendship Friend
        {
            get;
            set;
        }


        public static void StartProcessor()
        {
            StartProcessor(QueueProcessor_ProcessMessage);
        }

        static void QueueProcessor_ProcessMessage(FriendshipChangeResolver e)
        {
            
            var userRegion = UserRegion.GetLocal(e.UserID);
            if (userRegion == null)
            {
                Logger.Warn("Unable to get user region. This should be impossible!", e);
                return;
            }

            var regionID = userRegion.RegionID;

            Action<ClientEndpoint> mobileAction = null;

            // Special case for friend requests
            if (e.InitiatingUserID != e.UserID)
            {
                if (e.Friend.Status == FriendshipStatus.Confirmed)
                {
                    mobileAction = ep => PushNotificationWorker.FriendChange(regionID, ep, PushNotificationType.FriendConfirmation, e.Friend.OtherUserID, e.UserID);
                }
                else if (e.Friend.Status == FriendshipStatus.AwaitingMe)
                {
                    mobileAction = ep => PushNotificationWorker.FriendChange(regionID, ep, PushNotificationType.FriendRequest, e.Friend.OtherUserID, e.UserID);                    
                }
            }

            var userStats = UserStatistics.GetByUserOrDefault(e.Friend.OtherUserID);

            // Get the conversation
            var privateConversation = PrivateConversation.GetByUserIDAndOtherUserID(e.Friend.UserID, e.Friend.OtherUserID);

            var externalCommunity = string.IsNullOrEmpty(userStats.WatchingChannelID) ? null : ExternalCommunity.GetByTwitchID(userStats.WatchingChannelID);

            // New notification type
            var friendshipContract = e.Friend.ToNotification(userStats, privateConversation, externalCommunity);

            ClientEndpoint.DispatchNotification(e.UserID, ep => FriendshipChangeNotifier.Create(ep, friendshipContract), mobileAction);
        }

    }
}
;