﻿using Curse.CloudQueue;
using Curse.Friends.Enums;
using Curse.Friends.Statistics;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Data.Models;

namespace Curse.Friends.Data
{
    [CloudQueueProcessor(4, true)]
    public class FriendshipSuggestionWorker : BaseCloudQueueWorkerMessage<FriendshipSuggestionWorker>
    {
        private static readonly HashSet<int> PersonsOfInterest = new HashSet<int>(new[] { 1, 817369 });
        private static readonly new LogCategory Logger = new LogCategory("FriendshipSuggestionWorker") { ReleaseLevel = LogLevel.Trace };
        private static readonly LogCategory ThrottledLogger = new LogCategory("FriendshipSuggestionWorker") { Throttle = TimeSpan.FromMinutes(1) };

        /// <summary>
        /// The ID of the receiving user
        /// </summary>
        public int UserID
        {
            get;
            set;
        }

        public bool NotifyAllSuggestions
        {
            get;
            set;
        }


        public static void StartProcessor()
        {
            StartProcessor(QueueProcessor_ProcessMessage);
        }

        private static FriendSuggestion[] FindNewSuggestions(int regionID, int userID, int suggestionsLimit, IEnumerable<int> existingSuggestionIDs, int[] allFriendIDs, int[] confirmedFriendIDs, int[] blockedUserIDs)
        {            
            var suggestions = new Dictionary<int, FriendSuggestion>();
            var excludeIDs = new HashSet<int>(allFriendIDs.Concat(existingSuggestionIDs).Concat(blockedUserIDs));            
            excludeIDs.Add(userID);
            
            if (PersonsOfInterest.Contains(userID))
            {
                Logger.Trace("Performing mutual friend search for a person of interest: " + userID,
                    new
                    {
                        regionID,
                        suggestionsLimit,
                        excludeIDs,
                        allFriendIDs,
                        confirmedFriendIDs
                    });
            }


            var mutualFriendsDict = new Dictionary<int, int>();

            foreach (var myFriendID in confirmedFriendIDs)
            {
                var friendFriends = UserStatistics.GetByUserOrDefault(myFriendID).FriendIDs;

                foreach (var friendFriend in friendFriends)
                {
                    if (excludeIDs.Contains(friendFriend))
                    {
                        continue;
                    }

                    // Get the target friend's statistics
                    var friendFriendCount = UserStatistics.GetByUserOrDefault(myFriendID).FriendCount;

                    // Exclude this friend if they have reached the limit
                    if (friendFriendCount >= Friendship.MaxFriendCount)
                    {
                        excludeIDs.Add(friendFriend);                        
                        continue;
                    }

                    if (mutualFriendsDict.ContainsKey(friendFriend))
                    {
                        ++mutualFriendsDict[friendFriend];
                    }
                    else if(mutualFriendsDict.Count < suggestionsLimit)
                    {
                        mutualFriendsDict.Add(friendFriend, 1);
                    }                    
                }                                                              
            }

            var friendCount = confirmedFriendIDs.Count();
            var minimumMutualFriends = 1;
            
            if (friendCount >= 10)
            {
                minimumMutualFriends = 3;
            }
            else if (friendCount >= 2)
            {
                minimumMutualFriends = 2;
            }

            foreach (var kvp in mutualFriendsDict.Where(p => p.Value >= minimumMutualFriends))
            {
                // Separate good and bad hints - creates a dictionary of bool,FriendHint[] where true is a bad hint and false is a good hint
                var hints = GetSanitizeFriendHints(kvp.Key);

                var hint = hints.FirstOrDefault(h => h.Type == FriendHintType.Username);
                if (hint == null)
                {
                    FriendHintSearchIndexer.CreateForUser(kvp.Key);
                    ThrottledLogger.Warn("Unable to find a username friend hint, their search data will be re-indexed.", new { kvp.Key });
                    continue;
                }

                suggestions.Add(kvp.Key, new FriendSuggestion
                {
                    UserID = userID,
                    OtherUserID = kvp.Key,
                    FriendCount = kvp.Value,
                    AvatarUrl = hint.AvatarUrl,
                    Username = hint.SearchTerm,
                    Status = FriendSuggestionStatus.Pending,
                    DateSuggested = DateTime.UtcNow
                });
            }
               
            if (PersonsOfInterest.Contains(userID))
            {
                Logger.Trace("Finished finding mutual friend suggestions for user: " + userID, suggestions.Values);
            }
            
            foreach (var s in suggestions.Values)
            {                
                s.Insert(regionID);
                FriendsStatsManager.Current.MutualFriendSuggestions.Track();                
            }

            return suggestions.Values.ToArray();           
        }

        static FriendHint[] GetSanitizeFriendHints(int userID)
        {
            var allHints = FriendHint.GetAllLocal(p => p.UserID, userID);

            try
            {
                var badHints = allHints.Where(p => string.IsNullOrEmpty(p.SearchTerm));
                foreach (var hint in badHints)
                {
                    Logger.Info("Deleting invalid friend hint", hint);
                    hint.Delete();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to cleanup invalid friend hints");
            }

            return allHints.Where(p => !string.IsNullOrEmpty(p.SearchTerm)).ToArray();

        }

        static FriendSuggestion[] GetSanitizedSuggestions(int regionID, int userID)
        {
            var allSuggestions = FriendSuggestion.GetAll(regionID, p => p.UserID, userID);

            try
            {
                var badSuggestions = allSuggestions.Where(p => string.IsNullOrEmpty(p.Username));
                foreach (var suggestion in badSuggestions)
                {
                    Logger.Info("Deleting invalid friend suggestion", suggestion);
                    suggestion.Delete();
                }
            }
            catch (Exception ex)
            {                
                Logger.Error(ex, "Failed to cleanup invalid suggestions");
            }

            return allSuggestions.Where(p => !string.IsNullOrEmpty(p.Username)).ToArray();

        }

        static void QueueProcessor_ProcessMessage(FriendshipSuggestionWorker e)
        {
            if (PersonsOfInterest.Contains(e.UserID))
            {
                Logger.Trace("Processing finding mutual friend suggestions for user: " + e.UserID, e);
            }

            var myRegion = UserRegion.GetLocal(e.UserID);
            if(myRegion == null)
            {
                return;
            }

            var user = myRegion.GetUser();
            if (user == null)
            {
                return;
            }

            // Separate good and bad suggestions - creates a dictionary of bool,FriendSuggestion[] where true is a bad suggestion and false is a good suggestion
            var existingSuggestions = GetSanitizedSuggestions(myRegion.RegionID, e.UserID);

            var pendingSuggestions = existingSuggestions.Where(p => p.Type == FriendSuggestionType.MutualFriend && p.Status == FriendSuggestionStatus.Pending).ToArray();

            // We need to include all friend IDs here, not just confirmed.
            var myFriends = Friendship.GetAll(myRegion.RegionID, p => p.UserID, e.UserID);
            var myFriendIDs = myFriends.Select(p => p.OtherUserID).ToArray();
            var myFriendCount = myFriends.Count(p => p.Status == FriendshipStatus.Confirmed || p.Status == FriendshipStatus.AwaitingThem);

            if (myFriendCount >= Friendship.MaxFriendCount)
            {                
                return;
            }

            var myConfirmedFriendIDs = myFriends.Where(p => p.Status == FriendshipStatus.Confirmed).Select(p => p.OtherUserID).ToArray();

            // Update the mutual friend count
            foreach (var suggestion in pendingSuggestions)
            {
                try
                {                    
                    var newFriendCount = User.GetMutualFriendIDs(e.UserID, suggestion.OtherUserID).Count();
                    if(suggestion.FriendCount != newFriendCount)
                    {
                        suggestion.FriendCount = newFriendCount;
                        suggestion.Update(p => p.FriendCount);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to recalculate mutual friend count");
                }
            }

            var mutualFriendSuggestionLimit = 20 - pendingSuggestions.Count(p => p.Type == FriendSuggestionType.MutualFriend);
            var newSuggestions = new FriendSuggestion[0];

            // If they haven't been suggested friends in at least a day, find new ones
            if (mutualFriendSuggestionLimit > 0)
            {
                var blockedUserIDs = UserBlock.GetAllForUser(myRegion.UserID).Select(p => p.OtherUserID).ToArray();
                newSuggestions = FindNewSuggestions(myRegion.RegionID, user.UserID, mutualFriendSuggestionLimit, existingSuggestions.Select(p => p.OtherUserID).ToArray(), myFriendIDs, myConfirmedFriendIDs, blockedUserIDs);                
            }

            FriendSuggestion[] finalSuggestions;

            // If we want all suggestions we need to combine them
            if(e.NotifyAllSuggestions)
            {
                var platformSuggestions = existingSuggestions.Where(p => p.Status == FriendSuggestionStatus.Pending
                                                                         &&
                                                                         (p.Type == FriendSuggestionType.GameFriend ||
                                                                          p.Type == FriendSuggestionType.PlatformFriend))
                    .OrderByDescending(p => p.FriendCount)
                    .Take(50);

                finalSuggestions = platformSuggestions.Concat(pendingSuggestions.Concat(newSuggestions)).ToArray();
            }
            else
            {
                finalSuggestions = newSuggestions;
            }

            // We have no suggestions!
            if (!finalSuggestions.Any())
            {
                return;
            }

            // Take the top 50 friends, by the friend count
            var suggestedFriends = FriendSuggestion.GetValidFriendSuggestions(e.UserID, finalSuggestions).OrderByDescending(p => p.Type).ThenByDescending(p => p.FriendCount).Take(50).ToArray();

            // Send a notificiation to each of this user's connected endpoints
            foreach(var endpoint in ClientEndpoint.GetAllConnected(e.UserID))
            {
                FriendSuggestionNotifier.Create(endpoint, suggestedFriends);
            }
 
        }

    }
}
