﻿using System;
using System.Collections.Generic;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;

namespace Curse.Friends.Data.Queues
{
    public class GroupChangeCoordinator : BaseGroupEndpointNotification<GroupChangeCoordinator>
    {
        public GroupChangeCoordinator() { }

        public GroupChangeCoordinator(Group endpoint) : base(endpoint)
        {

        }

        public GroupChangeType ChangeType { get; set; }

        public int SenderID { get; set; }

        public HashSet<int> UserIDs { get; set; }

        public Guid[] AffectedGroupIDs { get; set; }

        public GroupMemberRemovedReason Reason { get; set; }

        public string MessageToUsers { get; set; }

        public GroupPresenceContract PresenceUpdate { get; set; }


        //The following methods enqueues based on the type of change requested for an existing group
        public static void AddUsers(Group group, int senderID, HashSet<int> userIDs)
        {
            GroupHostManager.EnsureServiceHost(group.RootGroup);

            new GroupChangeCoordinator(group.RootGroup)
            {
                UserIDs = userIDs,
                ChangeType = GroupChangeType.AddUsers,
                SenderID = senderID,
                AffectedGroupIDs = new[] { group.GroupID }

            }.Enqueue();

        }

        public static void RemoveUsers(Group group, int senderID, HashSet<int> userIDs, GroupMemberRemovedReason reason, string message)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    UserIDs = userIDs,
                    ChangeType = GroupChangeType.RemoveUsers,
                    SenderID = senderID,
                    AffectedGroupIDs = new[] { group.GroupID },
                    Reason = reason,
                    MessageToUsers = message
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of removed users!");
            }
        }

        public static void ChangeInfo(Group group, int senderID)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    ChangeType = GroupChangeType.ChangeInfo,
                    SenderID = senderID,
                    AffectedGroupIDs = new[] { group.GroupID },
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of removed users!");
            }

        }

        public static void UpdateUsers(Group group, int senderID, HashSet<int> userIDs)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    ChangeType = GroupChangeType.UpdateUsers,
                    SenderID = senderID,
                    AffectedGroupIDs = new[] { group.GroupID },
                    UserIDs = userIDs,
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of removed users!");
            }

        }

        public static void UpdateUserPresence(Group group, GroupPresenceContract groupPresence)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    ChangeType = GroupChangeType.UpdateUserPresence,
                    PresenceUpdate = groupPresence
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of removed users!");
            }

        }

        public static void VoiceSessionChanged(Group group, HashSet<int> userIDs, string callCode, GroupChangeType type)
        {
            GroupHostManager.EnsureServiceHost(group.RootGroup);

            new GroupChangeCoordinator(group.RootGroup)
            {
                AffectedGroupIDs = new[] { group.GroupID },
                ChangeType = type,
                UserIDs = userIDs,
            }.Enqueue();
        }

        public static void CreateGroup(Group group, int senderID)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    AffectedGroupIDs = new[] { group.GroupID },
                    ChangeType = GroupChangeType.CreateGroup,
                    SenderID = senderID,
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of a created group!");
            }
        }

        public static void RemoveGroup(Group group, int senderID)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    AffectedGroupIDs = new[] { group.GroupID },
                    ChangeType = GroupChangeType.RemoveGroup,
                    SenderID = senderID,
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of a removed group!");
            }

        }

        public static void UpdateEmoticons(Group group, int senderID)
        {
            try
            {
                GroupHostManager.EnsureServiceHost(group.RootGroup);

                new GroupChangeCoordinator(group.RootGroup)
                {
                    ChangeType = GroupChangeType.UpdateEmoticons,
                    SenderID = senderID
                }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group service of an emoticon change.");
            }
        }
    }
}
