﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Extensions;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;

namespace Curse.Friends.Data.Queues
{
    public enum GroupMemberIndexWorkerType
    {
        MembersAdded,
        MembersRemoved,
        MemberRoleAdded,
        MemberRoleRemoved,
        Complete,
        RoleRanks,
        MembersActivity,
        MemberRenamed
    }

    public class GroupMemberIndexWorker : BaseCloudQueueShoveledMessage<GroupMemberIndexWorker>
    {
        public GroupMemberIndexWorkerType Type { get; set; }

        public Guid GroupID { get; set; }

        public int GroupRegionID { get; set; }

        public DateTime ActivityTimestamp { get; set; }

        public int[] AffectedUserIDs { get; set; }

        public int RoleID { get; set; }

        public Dictionary<int, Tuple<int, int>> MemberBestRoles { get; set; }

        public Dictionary<int, string> Renames { get; set; }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static bool ShouldIndex(Group group)
        {
            if (group.Type != GroupType.Large)
            {
                return false;
            }

            if (!group.IsRootGroup)
            {
                return false;
            }

            return true;
        }

        public static void CreateMembersAddedWorker(Group group, HashSet<int> addedUserIDs)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                AffectedUserIDs = addedUserIDs.ToArray(), 
                Type = GroupMemberIndexWorkerType.MembersAdded
            }.Enqueue();
        }

        public int UserID { get; set; }
        public string Username { get; set; }
        public string Nickname { get; set; }
        public string DisplayName { get; set; }

        public static void CreateMemberNicknameWorker(Group group, int userID, string username, string nickname, string displayName)
        {
            if (!ShouldIndex(group))
            {
                return;
            }

            new GroupMemberIndexWorker
            {
                Type = GroupMemberIndexWorkerType.MemberRenamed,
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                UserID = userID, 
                Username = username,
                Nickname = nickname,
                DisplayName = displayName
            }.Enqueue();
        }

        public static void CreateMembersRemovedWorker(Group group, HashSet<int> removedMemberUserIDs)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                AffectedUserIDs = removedMemberUserIDs.ToArray(),
                Type = GroupMemberIndexWorkerType.MembersRemoved
            }.Enqueue();
        }

        public static void CreateRoleAddedWorker(Group group, int affectedUserID, GroupRole newRole)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                AffectedUserIDs = new[] {affectedUserID},
                RoleID = newRole.RoleID, 
                Type = GroupMemberIndexWorkerType.MemberRoleAdded
            }.Enqueue();
        }

        public static void CreateRoleRemovedWorker(Group group, int affectedUserID, GroupRole role)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                AffectedUserIDs = new[] {affectedUserID},
                Type = GroupMemberIndexWorkerType.MemberRoleRemoved,
                RoleID = role.RoleID
            }.Enqueue();
        }

        public static void CreateFullSync(Group group)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                Type = GroupMemberIndexWorkerType.Complete,
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID
            }.Enqueue();
        }

        public static void RoleRanks(Group group, Dictionary<int, Tuple<int, int>> newBestRoles)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                Type = GroupMemberIndexWorkerType.RoleRanks,
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                MemberBestRoles = newBestRoles
            }.Enqueue();
        }

        public static void CreateMembersActivity(Group group, int[] userIDs, DateTime timestamp)
        {
            if (!ShouldIndex(group))
            {
                return;
            }
            
            new GroupMemberIndexWorker
            {
                Type = GroupMemberIndexWorkerType.MembersActivity,
                ActivityTimestamp = timestamp,
                GroupID = group.GroupID,
                GroupRegionID = group.RegionID,
                AffectedUserIDs = userIDs
            }.Enqueue();
        }


        private static void Process(GroupMemberIndexWorker worker)
        {
            try
            {                
                var group = Group.Get(worker.GroupRegionID, worker.GroupID);
                
                if (group == null)
                {                    
                    Logger.Warn("Group was not found while processing a member index job.");
                    return;
                }

                if (group.Type != GroupType.Large)
                {
                    Logger.Warn("Attempted to index a Normal group.");
                    return;
                }

                switch (worker.Type)
                {
                    case GroupMemberIndexWorkerType.MemberRoleAdded:
                    case GroupMemberIndexWorkerType.MemberRoleRemoved:
                    case GroupMemberIndexWorkerType.MembersAdded:
                    {
                        var affectedMembers = GroupMember.MultiGet(worker.GroupRegionID, worker.AffectedUserIDs.Select(id => new KeyInfo(worker.GroupID, id)));
                        foreach (var member in affectedMembers)
                        {
                            try
                            {
                                GroupMemberManager.AddOrUpdateMember(member);
                            }
                            catch (Exception ex)
                            {
                                Logger.Trace(ex, "Failed to add or update member to index", new {worker, member});
                            }
                        }
                        break;
                    }
                    case GroupMemberIndexWorkerType.MembersRemoved:
                    {                        
                        var affectedMembers = GroupMember.MultiGet(worker.GroupRegionID,
                            worker.AffectedUserIDs.Select(id => new KeyInfo(worker.GroupID, id)));
                        foreach (var member in affectedMembers)
                        {
                            try
                            {
                                GroupMemberManager.DeleteMember(member);
                            }
                            catch (Exception ex)
                            {
                                Logger.Trace(ex, "Failed to remove member from index", new {worker, member});
                            }
                        }
                        
                        break;
                    }
                    case GroupMemberIndexWorkerType.RoleRanks:
                    {
                        foreach (var list in worker.MemberBestRoles.Keys.InSetsOf(100))
                        {
                            var affectedMembers = GroupMember.MultiGet(worker.GroupRegionID, list.Select(id => new KeyInfo(worker.GroupID, id)));
                            var searchModels = new List<GroupMemberSearchModel>();

                            foreach (var member in affectedMembers)
                            {
                                Tuple<int, int> roleAndRank;
                                if (!worker.MemberBestRoles.TryGetValue(member.UserID, out roleAndRank))
                                {
                                    Logger.Warn("Unable to process role rank change. The member was not found in the role dictionary.");
                                    continue;
                                }

                                // Ensure there's not a replication lag related issue, by updating the values here before indexing.
                                member.BestRole = roleAndRank.Item1;
                                member.BestRoleRank = roleAndRank.Item2;

                                searchModels.Add(GroupMemberSearchModel.FromGroupMember(member));                             
                            }

                            GroupMemberManager.BatchUpsertMember(searchModels);     
                        }
                        break;    
                    }
                    case GroupMemberIndexWorkerType.MembersActivity:
                    {
                        // Update the elastic index
                        foreach (var batch in worker.AffectedUserIDs.InSetsOf(500))
                        {
                            GroupMemberManager.UpdateActivityDate(worker.GroupID, batch.ToArray(), worker.ActivityTimestamp);
                        }
                        break;
                    }                  
                    case GroupMemberIndexWorkerType.Complete:
                    {                        
                        group.DateMembersIndexed = DateTime.UtcNow;
                        group.Update(g => g.DateMembersIndexed);

                        // Need all members, including deleted ones
                        var allMembers = group.TryGetAllMembers();
                        
                        if (allMembers == null)
                        {
                            Logger.Warn("Unable to complete complete member indexing. Failed to retrieve all group members from database.");
                            return;
                        }

                        var members = allMembers.ToDictionary(g => g.UserID);
                        var searchModels = members.Values.Select(GroupMemberSearchModel.FromGroupMember).ToArray();
                        foreach (var batch in searchModels.InSetsOf(500))
                        {
                            GroupMemberManager.BatchUpsertMember(batch);
                        }
                        break;
                    }
                    case GroupMemberIndexWorkerType.MemberRenamed:
                    {                       
                        // Update the elastic index
                        GroupMemberManager.UpdateUsername(worker.GroupID, worker.UserID, worker.Username, worker.Nickname, worker.DisplayName);                        
                        
                        break;    
                    }
                    default:
                        throw new NotImplementedException("GroupMemberIndexWorkerType not supported: " + worker.Type);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while processing group member index worker", worker);
            }
        }
    }
}
