﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.UI;
using Aerospike.Client;
using Curse.CloudQueue;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Newtonsoft.Json;

namespace Curse.Friends.Data.Queues
{
    public enum GroupMemberListWorkerType
    {
        Created,
        AddUsers,
        RemoveUsers,
        ChangeRoles,
        ChangeCurrentGroup,
        DeleteGroup,
        CallStarted,
        CallEnded
    }

    /// <summary>
    /// Responsible for replicating group membership changes to remote regions
    /// </summary>    
    [CloudQueueProcessor(4, true)]
    public class GroupMemberListWorker : BaseCloudQueueShoveledMessage<GroupMemberListWorker>
    {
        public Guid GroupID { get; set; }
        
        public NewGroupMember[] Members { get; set; }

        public GroupMemberListWorkerType Type { get; set; }

        public Guid CurrentGroupID { get; set; }

        public Guid[] AffectedGuids { get; set; }

        [JsonConstructor]
        public GroupMemberListWorker() { }

        protected GroupMemberListWorker(int destinationRegionID) : base(destinationRegionID)
        {
            
        }

        public static void Create(int destinationRegionID, GroupMemberListWorkerType type, Guid groupID, NewGroupMember[] members)
        {
            new GroupMemberListWorker(destinationRegionID)
            {                
                GroupID = groupID,
                Members = members,                
                Type = type
            }.Enqueue();
        }

        public static void CreateCurrentGroupChanged(int destinationRegionID, Guid groupID, NewGroupMember[] members, int sourceRegionID, Guid currentGroupID)
        {
            new GroupMemberListWorker(destinationRegionID)
            {                
                GroupID = groupID,
                CurrentGroupID = currentGroupID,
                Members = members,             
                Type = GroupMemberListWorkerType.ChangeCurrentGroup
            }.Enqueue();
        }

        /// <summary>
        /// Creates worker to delete root group or channel. 
        /// </summary>
        public static void CreateGroupDeleteWorker(int destinationRegionID, Guid groupID, Guid[] groupIDs)
        {
            new GroupMemberListWorker(destinationRegionID)
            {                
                GroupID = groupID,
                Type = GroupMemberListWorkerType.DeleteGroup,
                AffectedGuids = groupIDs
            }.Enqueue();
        }

        /// <summary>
        /// Creates worker to delete root group or channel. 
        /// </summary>
        public static void CreateActiveCallWorker(int destinationRegionID, Guid groupID, int[] memberIDs, bool isActive)
        {
            new GroupMemberListWorker(destinationRegionID)
            {                
                GroupID = groupID,
                Type = isActive ? GroupMemberListWorkerType.CallStarted : GroupMemberListWorkerType.CallEnded,
                Members = memberIDs.Select(p => new NewGroupMember(p)).ToArray()
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(QueueProcessor_ProcessMessage);
        }

        private static void QueueProcessor_ProcessMessage(GroupMemberListWorker e)
        {
            // Try to get the member list            
            switch (e.Type)
            {
                case GroupMemberListWorkerType.Created:
                    CopyGroup(e.GroupID, e.SourceRegionID);
                    break;

                case GroupMemberListWorkerType.AddUsers:
                    {
                        var memberList = GroupMemberList.GetLocal(e.GroupID);
                        if (memberList == null)
                        {
                            CopyGroup(e.GroupID, e.SourceRegionID);
                        }
                        else
                        {
                            foreach (var newMember in e.Members)
                            {
                                if (memberList.Members.Exists(newMember.UserID))
                                {
                                    continue;                                    
                                }

                                var gm = new LegacyGroupMember
                                {
                                    RegionID = newMember.RegionID,
                                    Role = newMember.Role,
                                    Username = newMember.Username,
                                    UserID = newMember.UserID,
                                    DateJoined = DateTime.UtcNow,
                                    DateLastActive = DateTime.UtcNow,
                                };

                                memberList.Members.Add(gm);
                            }
                        }
                        break;
                    }
                case GroupMemberListWorkerType.RemoveUsers:
                    {

                        var group = Group.GetLocal(e.GroupID);
                        if (group == null)
                        {
                            Logger.Warn("Unable to remove group members. Group could not be retrieved from local region.", e);
                            return;
                        }

                        var memberList = group.MemberList;
                        if (memberList == null)
                        {
                            Logger.Warn("Unable to remove group members. Local member list is null.", e);
                            return;
                        }

                        foreach (var member in e.Members)
                        {
                            memberList.Members.Remove(member.UserID);
                        }

                        break;
                    }
                case GroupMemberListWorkerType.ChangeRoles:
                    {
                        var group = Group.GetLocal(e.GroupID);
                        if (group == null)
                        {
                            Logger.Warn("Unable to change group roles for members. Group could not be retrieved from local region.", e);
                            return;
                        }

                        var memberList = group.MemberList;
                        if (memberList == null)
                        {
                            Logger.Warn("Unable to change roup roles for members. Local member list is null.", e);
                            return;
                        }

                        foreach (var member in e.Members)
                        {
                            var existing = memberList.Members.Find(member.UserID);
                            if (existing != null)
                            {
                                existing.Role = member.Role;
                                memberList.Members.Update(existing);
                            }
                            else if (group.IsRemote)
                            {
                                Logger.Info("ChangeRoles - Local group member is missing, so copying remote group member!");
                                CopyRemoteGroupMember(group.GroupID, group.RegionID, member.UserID);
                            }
                            else
                            {
                                Logger.Warn("Unable to process change role for member. They were not found in the group member list.", new { member.UserID, e.GroupID });
                            }
                        }

                        break;
                    }
                case GroupMemberListWorkerType.ChangeCurrentGroup:
                    {
                        if (e.Members == null)
                        {
                            Logger.Warn("Unable to change current group for members. Supplied member list is null.", e);
                            return;
                        }

                        var group = Group.GetLocal(e.GroupID);
                        if (group == null)
                        {
                            Logger.Warn("Unable to change current group for members. Group could not be retrieved from local region.", e);
                            return;
                        }

                        var memberList = group.MemberList;
                        if (memberList == null)
                        {
                            Logger.Warn("Unable to change current group for members. Local member list is null.", e);
                            return;
                        }

                        foreach (var m in e.Members)
                        {
                            var member = memberList.Members.Find(m.UserID);
                            if (member != null)
                            {
                                member.CurrentGroupID = e.CurrentGroupID;
                                memberList.Members.Update(member);
                            }

                            else if(group.IsRemote)
                            {
                                Logger.Info("ChangeCurrentGroup - Local group member is missing, so copying remote group member!");
                                CopyRemoteGroupMember(group.GroupID, group.RegionID, m.UserID);                                
                            }
                            else
                            {
                                Logger.Warn("Unable to process change for member. They were not found in the group member list.", new { m.UserID, e.GroupID });
                            }
                        }
                        break;
                    }
                case GroupMemberListWorkerType.DeleteGroup:
                    {
                        //deletes all subchannels/children of the requested GroupID
                        foreach (var child in e.AffectedGuids)
                        {
                            GroupMemberList.DeleteByKeyLocal(child);
                        }

                        GroupMemberList.DeleteByKeyLocal(e.GroupID);
                    }
                    break;

                case GroupMemberListWorkerType.CallEnded:
                case GroupMemberListWorkerType.CallStarted:
                    UpdateCallInfo(e.GroupID, e.Members, e.Type == GroupMemberListWorkerType.CallStarted);
                    break;

                default:
                    throw new ArgumentException("Unknown type: " + e.Type);
            }

        }

        private static void UpdateCallInfo(Guid groupID, NewGroupMember[] members, bool isCallActive)
        {
            foreach (var member in members)
            {
                var membership = GroupMembership.GetLocalByUserAndGroupID(member.UserID, groupID);
                if (membership == null)
                {
                    // Check the memberlist, see if we have a race condition
                    var rootGroup = Group.GetLocal(groupID).RootGroup;
                    if (rootGroup == null)
                    {
                        Logger.Warn("UpdateCallInfo - Unknown root group!");
                        continue;
                    }

                    if (rootGroup.MemberList.Members.Find(member.UserID) == null)
                    {
                        Logger.Warn("UpdateCallInfo - GroupMembership is missing and so is the member list!", new { member });
                    }
                    else
                    {
                        Logger.Warn("UpdateCallInfo - GroupMembership is missing, but there is a member list entry.", new { member });
                    }
                    
                    continue;
                }
                if (membership.IsCallActive != isCallActive)
                {
                    membership.IsCallActive = isCallActive;
                    membership.Update(p => p.IsCallActive);
                }                
            }            
        }

        private static void CopyGroup(Guid groupID, int sourceRegionID)
        {
            var remoteList = GroupMemberList.Get(sourceRegionID, groupID);
            var localList = GroupMemberList.GetLocal(groupID);
            if (localList == null)
            {
                localList = new GroupMemberList { GroupID = groupID };
                localList.InsertLocal();
            }

            var allMembers = remoteList.Members.GetValues();
            foreach (var member in allMembers)
            {
                localList.Members.Add(member);
            }

        }

        private static void CopyRemoteGroupMember(Guid groupID, int sourceRegionID, int memberID)
        {
            var remoteList = GroupMemberList.Get(sourceRegionID, groupID);
            var localList = GroupMemberList.GetLocal(groupID);
            if (localList == null)
            {
                localList = new GroupMemberList { GroupID = groupID };
                localList.InsertLocal();
            }

            var remoteMember = remoteList.Members.Find(memberID);
            if (remoteMember == null)
            {
                Logger.Warn("Unable to copy remote group member. It could not be retrieved!", new { groupID, sourceRegionID, memberID });
                return;
            }

            localList.Members.Add(remoteMember);            

        }
    }
}
