﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudQueue;
using Curse.Friends.Data.DerivedModels;
using Newtonsoft.Json;

namespace Curse.Friends.Data
{
    public enum GroupMemberWorkerType
    {
        Info,
        NewGroup,
        ChangeAccessLevel,
        ChangeMemberRole,
        DeleteGroup,
        DeleteChannel,
        RestoreChannel,
        VoiceSessionChanged,
        Restructured,        
        ChangeRoleRank,
        MemberAcitivity,
        MembersRemoved,
        RoleDeleted,
        MembersAdded,
        AccountUnlinked,
    }

    [CloudWorkerQueue(0)]
    [CloudQueueProcessor(4, true)]
    public class GroupMemberWorker : BaseCloudQueueShoveledMessage<GroupMemberWorker>
    {
        [JsonConstructor]
        public GroupMemberWorker() { }

        public GroupMemberWorker(Group group)
            : base(group.RegionID)
        {

        }

        public Guid GroupID { get; set; }

        public GroupMemberWorkerType Type { get; set; }

        public Guid ChildGroupID { get; set; }

        public NewGroupMember[] NewMembers { get; set; }

        public int RequestingUserID { get; set; }

        public int AffectedUserID { get; set; }

        public int AffectedRoleID { get; set; }

        public Guid[] AffectedGroupIDs { get; set; }

        public Guid VoiceSessionID { get; set; }

        public HashSet<int> AffectedUserIDs { get; set; }


        /// <summary>
        /// updates the avatarUrl and grouptitle for membership
        /// </summary>
        public static void CreateGroupChangeWorker(Group rootGroup, Group affectedGroup)
        {
            new GroupMemberWorker(rootGroup)
            {
                Type = GroupMemberWorkerType.Info,
                GroupID = rootGroup.GroupID,
                ChildGroupID = affectedGroup.GroupID,
            }.Enqueue();
        }


        /// <summary>
        /// Creates group membership records for all existing members of a root group's new sub group (as long as they meet the minimum role reqs)
        /// </summary>
        public static void CreateNewGroupWorker(Group group)
        {
            new GroupMemberWorker(group.RootGroup)
            {
                Type = GroupMemberWorkerType.NewGroup,
                GroupID = group.RootGroupID,
                ChildGroupID = group.GroupID,
            }.Enqueue();
        }

        /// <summary>
        /// Deletes all groupmembership records related to the group and also any child groups if exists
        /// Can be called when a root group or any individual sub channel is deleted
        /// </summary>
        public static void CreateDeleteGroupWorker(Group group)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.DeleteGroup,
                GroupID = group.GroupID,
            }.Enqueue();
        }

        /// <summary>
        /// Soft deletes all group member records related to the affected groups
        /// </summary>
        public static void CreateDeleteChannelWorker(Group rootGroup, Guid[] affectedGroupIDs)
        {
            if (!rootGroup.IsRootGroup)
            {
                throw new InvalidOperationException("CreateDeleteChannelWorker requires a root group.");
            }

            if (affectedGroupIDs == null || !affectedGroupIDs.Any())
            {
                throw new InvalidOperationException("CreateDeleteChannelWorker requires at least one child group.");
            }

            new GroupMemberWorker(rootGroup)
            {
                Type = GroupMemberWorkerType.DeleteChannel,
                GroupID = rootGroup.GroupID,
                AffectedGroupIDs = affectedGroupIDs
            }.Enqueue();
        }

        /// <summary>
        /// Soft undeletes all group member records related to the affected groups
        /// </summary>
        public static void CreateRestoreChannelWorker(Group group, Guid[] affectedGroupIDs)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.RestoreChannel,
                GroupID = group.GroupID,
                AffectedGroupIDs = affectedGroupIDs
            }.Enqueue();
        }
        public static void CreateGroupRestructuredWorker(Group group)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.Restructured,
                GroupID = group.GroupID,
            }.Enqueue();
        }

        public static void CreateRoleRank(Group group)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.ChangeRoleRank,
                GroupID = group.GroupID,
            }.Enqueue();
        }

        public static void CreateMemberActivity(Group group, HashSet<int> activeUserIDs)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.MemberAcitivity,
                GroupID = group.GroupID,
                AffectedUserIDs = activeUserIDs
            }.Enqueue();
        }

        public static void CreateMembersRemoved(Group group, IEnumerable<int> removedMemberIDs)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.MembersRemoved,
                GroupID = group.GroupID,
                AffectedUserIDs = new HashSet<int>(removedMemberIDs)
            }.Enqueue();
        }

        public static void CreateMembersAdded(Group group, HashSet<int> addedUserIDs)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.MembersAdded,
                GroupID = group.GroupID,
                AffectedUserIDs = addedUserIDs
            }.Enqueue();
        }


        public static void CreateRoleDeleted(Group group, int requestingUserID, int roleID)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.RoleDeleted,
                GroupID = group.GroupID,
                RequestingUserID = requestingUserID,
                AffectedRoleID = roleID
            }.Enqueue();
        }

        public static void CreateAccountUnlinked(Group group, int requestingUserID)
        {
            new GroupMemberWorker(group)
            {
                Type = GroupMemberWorkerType.AccountUnlinked,
                GroupID = group.GroupID,
                RequestingUserID = requestingUserID
            }.Enqueue();
        }
    }
}
