﻿using System;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;

namespace Curse.Friends.Data.Queues
{
    public enum GroupSearchIndexWorkerType
    {
        GroupCreated,
        GroupInfoUpdated,
        GroupMemberCountUpdated,
        GroupDeleted,
        GroupStreamingStatus,
        GroupFeaturedStatus,

        GroupQuarantine = 1000
    }

    public class GroupSearchIndexWorker : BaseCloudQueueShoveledMessage<GroupSearchIndexWorker>
    {
        public GroupSearchIndexWorkerType Type { get; set; }
        public Group Group { get; set; }
        public GroupSearchSettings GroupSearchSettings { get; set; }

        public static void CreateGroupCreatedWorker(Group group, GroupSearchSettings groupSearchSettings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupCreated,
                Group = group,
                GroupSearchSettings = groupSearchSettings
            }.Enqueue();

            Logger.Debug("New group queued: " + group.Title);
        }

        public static void CreateGroupInfoUpdatedWorker(Group group, GroupSearchSettings groupSearchSettings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupInfoUpdated,
                Group = group,
                GroupSearchSettings = groupSearchSettings
            }.Enqueue();

            Logger.Debug("Update group queued: " + group.Title);
        }

        public static void CreateGroupMemberCountUpdatedWorker(Group group, GroupSearchSettings groupSearchSettings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupMemberCountUpdated,
                Group = group,
                GroupSearchSettings = groupSearchSettings
            }.Enqueue();
        }

        public static void CreateGroupDeletedWorker(Group group, GroupSearchSettings groupSearchSettings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupDeleted,
                Group = group,
                GroupSearchSettings = groupSearchSettings
            }.Enqueue();
        }

        public static void CreateGroupStreamingStatus(Group group, GroupSearchSettings groupSearchSettings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupStreamingStatus,
                Group = group,
                GroupSearchSettings = groupSearchSettings
            }.Enqueue();
        }

        public static void CreateGroupFeaturedStatus(Group group, GroupSearchSettings groupSearchSettings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupFeaturedStatus,
                Group = group,
                GroupSearchSettings = groupSearchSettings
            }.Enqueue();
        }

        public static void CreateGroupQuarantineWorker(Group group, GroupSearchSettings settings)
        {
            new GroupSearchIndexWorker
            {
                Type = GroupSearchIndexWorkerType.GroupQuarantine,
                Group = group,
                GroupSearchSettings = settings,
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(GroupSearchIndexWorker worker)
        {
            try
            {
                
                var group = worker.Group;
                
                if (group == null)
                {
                    Logger.Warn("Unable to index group. Group or group search settings are null.");
                    return;
                }

                if (!group.IsRootGroup)
                {
                    Logger.Warn("Unable to index group. Group supplied is not the root group.");
                    return;
                }

                if (group.Type != GroupType.Large)
                {
                    Logger.Warn("Unable to index group. Group supplied is not a Large group.");
                    return;
                }


                var owner = group.GetMember(group.OwnerID, true);
                if (owner == null) // If it's missing, get it from the home region
                {
                    owner = group.GetMember(group.OwnerID);
                }
               
                Logger.Trace("Indexing group due to change", new { GroupTitle = group.Title, ChangeType = worker.Type });

                var linkedCommunities = ExternalCommunityMapping.GetAllLocal(c => c.GroupID, group.GroupID).Where(c => !c.IsDeleted);
                var communities = ExternalCommunity.MultiGetLocal(linkedCommunities.Select(c => new KeyInfo(c.ExternalID, c.Type)));
                var bestCommunity = communities.Count == 0 ? null : communities.OrderByDescending(c => c.Followers).First();

                switch (worker.Type)
                {
                    case GroupSearchIndexWorkerType.GroupCreated:
                    case GroupSearchIndexWorkerType.GroupInfoUpdated:
                    {
                        var model = GroupSearchModel.FromData(group, worker.GroupSearchSettings, owner, Avatar.GetByTypeAndID(AvatarType.Group, group.GroupID.ToString()),
                            Avatar.GetByTypeAndID(AvatarType.GroupCover, group.GroupID.ToString()), bestCommunity);
                        GroupSearchManager.Index(model);
                        break;
                    }
                    case GroupSearchIndexWorkerType.GroupDeleted:
                        GroupSearchManager.UpdateSearchability(group.GroupID, false);
                        break;
                    case GroupSearchIndexWorkerType.GroupMemberCountUpdated:
                        GroupSearchManager.UpdateMemberCount(group.GroupID, group.MemberCount,
                            GroupSearchModel.CalculateQuality(group, worker.GroupSearchSettings, Avatar.GetByTypeAndID(AvatarType.Group, group.GroupID.ToString()), bestCommunity));
                        break;
                    case GroupSearchIndexWorkerType.GroupStreamingStatus:
                        GroupSearchManager.UpdateStreamingStatus(group.GroupID, group.IsStreaming, group.StreamingTimestamp);
                        break;
                    case GroupSearchIndexWorkerType.GroupFeaturedStatus:
                        GroupSearchManager.UpdateFeaturedStatus(group.GroupID, group.IsFeatured, group.FeaturedTimestamp);
                        break;
                    case GroupSearchIndexWorkerType.GroupQuarantine:
                        GroupSearchManager.UpdateQuarantineStatus(group.GroupID, group.FlaggedAsInappropriate);
                        break;
                    default:
                        Logger.Warn("Unexpected worker type: " + worker.Type, worker);
                        return;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process worker", worker);
            }
        }
    }
}
