﻿using System;
using System.Collections.Concurrent;
using System.Threading;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.Data
{
    public class MessageLock<T> : IDisposable where T:BaseLockableMessage<T>
    {
        private static Func<T, bool> _unlockFunc;
 
        public static void Initialize(Func<T, bool> unlockFunc)
        {
            _unlockFunc = unlockFunc;
        }

        public static void ProcessRoleSyncCheck(T message)
        {
            MessageLock<T> lockObject;
            if (_locks.TryGetValue(message.RequestID, out lockObject))
            {
                lock (lockObject)
                {
                    if (_locks.ContainsKey(message.RequestID))
                    {
                        lockObject._count++;
                        var eligible = _unlockFunc(message);
                        if (eligible || lockObject._count == lockObject._total)
                        {
                            lockObject._isEligible |= eligible;
                            Monitor.Pulse(lockObject);
                        }
                    }
                }
            }
        }

        public static MessageLock<T> Create(Guid requestID, int count)
        {
            var lockManager = new MessageLock<T>(requestID, count);
            if (!_locks.TryAdd(requestID, lockManager))
            {
                throw new InvalidOperationException();
            }

            return lockManager;
        }


        private static readonly ConcurrentDictionary<Guid, MessageLock<T>> _locks = new ConcurrentDictionary<Guid, MessageLock<T>>();

        private readonly int _total;
        private readonly Guid _id;
        private int _count;
        private bool _isEligible;

        private MessageLock(Guid id, int total)
        {
            _id = id;
            _total = total;
            _count = 0;
        }

        public void IncrementCount(bool eligible=false)
        {
            _count++;
        }

        public bool WaitFor(TimeSpan timeout)
        {
            if (_count==_total)
            {
                return _isEligible;
            }

            return Monitor.Wait(this, timeout) && _isEligible;
        }

        public void Dispose()
        {
            MessageLock<T> counter;
            _locks.TryRemove(_id, out counter);
        }
    }
}