﻿using Curse.CloudQueue;
using Curse.Friends.Enums;
using Curse.Logging;
using System;

namespace Curse.Friends.Data
{
    [CloudWorkerQueue(0)]
    [CloudQueueProcessor(10, true, PrefetchMultiplier = 3.0)]
    [CloudQueue(true)]
    public class RegionalUserStatusResolver : BaseCloudQueueShoveledMessage<RegionalUserStatusResolver>
    {
        public new static readonly LogCategory Logger = new LogCategory("RegionalUserStatusResolver") { ReleaseLevel = LogLevel.Info };

        public RegionalUserStatusResolver() { }

        public RegionalUserStatusResolver(int regionID) : base(regionID)
        {

        }

        public static void CreateIdleResolver(int userID, int regionID, string machineKey)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                MachineKey = machineKey,
                DisconnectEndpoint = false,
                ResolverType = "Idle",
                SourceServerName = Environment.MachineName,
                Type = UserStatusResolver.UserStatusResolverType.Idle
            }.Enqueue();
        }

        public static void CreateActiveResolver(int userID, int regionID, string machineKey)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                MachineKey = machineKey,
                DisconnectEndpoint = false,
                ResolverType = "Active",
                SourceServerName = Environment.MachineName,
                Type = UserStatusResolver.UserStatusResolverType.Active
            }.Enqueue();
        }

        public static void CreateGameResolver(int userID, int regionID, string machineKey, bool isGameRunning, int gameID, int gameState, string gameStatusMessage)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                MachineKey = machineKey,
                DisconnectEndpoint = false,
                ResolverType = "Game",
                SourceServerName = Environment.MachineName,
                GameID = gameID,
                GameState = gameState,
                IsGameRunning = isGameRunning,
                GameStatusMessage = gameStatusMessage,
                Type = UserStatusResolver.UserStatusResolverType.Game
            }.Enqueue();
        }

        public static void CreateWatchingChannelResolver(int userID, int regionID, string machineKey, bool isWatching, string externalCommunityID)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                MachineKey = machineKey,
                DisconnectEndpoint = false,
                ResolverType = "Watching",
                SourceServerName = Environment.MachineName,
                ExternalCommunityID = externalCommunityID,
                IsWatching = isWatching,
                Type = UserStatusResolver.UserStatusResolverType.Channel
            }.Enqueue();
        }

        public UserStatusResolver.UserStatusResolverType Type { get; set; }

        /// <summary>
        /// When the user manually changes their status, or becomes idle.
        /// </summary>
        /// <param name="userID"></param>
        /// <param name="status"></param>
        /// <param name="machineKey"></param>
        /// <param name="sessionID"></param>
        public static void CreateStatusChangeResolver(int userID, int regionID, UserConnectionStatus status, string machineKey, string sessionID)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                Status = status,
                MachineKey = machineKey,
                SessionID = sessionID,
                DisconnectEndpoint = false,
                ResolverType = "Status Change Resolver",
                SourceServerName = Environment.MachineName,
                Type = UserStatusResolver.UserStatusResolverType.Connection
            }.Enqueue();
        }

        /// <summary>
        /// When the user disconnects from the notification server.
        /// </summary>
        /// <param name="endpoint"></param>        
        public static void CreateOfflineResolver(ClientEndpoint endpoint, int regionID)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = endpoint.UserID,
                MachineKey = endpoint.MachineKey,
                SessionID = endpoint.SessionID,
                Status = UserConnectionStatus.Offline,
                DisconnectEndpoint = false,
                ResolverType = "Offline Resolver",
                SourceServerName = Environment.MachineName,
                ConnectionID = endpoint.ConnectionID,
                Type = UserStatusResolver.UserStatusResolverType.Connection
            }.Enqueue();
        }

        /// <summary>
        /// When a user's session is not found, or the server starts up.        
        /// </summary>
        /// <param name="endpoint"></param>
        public static void CreateDisconnectResolver(ClientEndpoint endpoint, int regionID)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = endpoint.UserID,
                MachineKey = endpoint.MachineKey,
                SessionID = endpoint.SessionID,
                Status = UserConnectionStatus.Offline,
                DisconnectEndpoint = true,
                ResolverType = "Endpoint Disconnect Resolver",
                SourceServerName = Environment.MachineName,
                ConnectionID = endpoint.ConnectionID,
                Type = UserStatusResolver.UserStatusResolverType.Connection
            }.Enqueue();
        }

        public static void CreatePresenceResolver(int userID, int regionID, string machineKey)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                MachineKey = machineKey,
                Type = UserStatusResolver.UserStatusResolverType.Presence,
            }.Enqueue();
        }

        public static void CreateTwitchStatusResolver(int userID, int regionID, string machineKey, UserConnectionStatus status)
        {
            new RegionalUserStatusResolver(regionID)
            {
                UserID = userID,
                MachineKey = machineKey,
                Type = UserStatusResolver.UserStatusResolverType.TwitchStatus,
                Status = status
            }.Enqueue();
        }

        public bool DisconnectEndpoint { get; set; }

        /// <summary>
        /// The effective status to use with the noticiation (Required when ResolveStatus is false)
        /// </summary>
        public UserConnectionStatus Status
        {
            get;
            set;
        }

        /// <summary>
        /// Used when resolving the user status (Optional)
        /// </summary>
        public string MachineKey
        {
            get;
            set;

        }

        /// <summary>
        /// Used when resolving the user status (Optional)
        /// </summary>
        public string SessionID
        {
            get;
            set;

        }

        /// <summary>
        /// The ID of the user who's status is changing.
        /// </summary>
        public int UserID
        {
            get;
            set;
        }

        public string ConnectionID
        {
            get;
            set;
        }

        public string SourceServerName
        {
            get; set;
        }

        public string ResolverType
        {
            get; set;
        }

        public int GameID { get; set; }

        public bool IsGameRunning { get; set; }

        public int GameState { get; set; }

        public string GameStatusMessage { get; set; }

        public string CustomStatusMessage { get; set; }

        public string ExternalCommunityID { get; set; }

        public bool IsWatching { get; set; }

    }


}
