﻿using System;
using Curse.CloudQueue;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Newtonsoft.Json;

namespace Curse.Friends.Data
{
    /// <summary>
    /// sends push notifications to various mobile device platforms
    /// </summary>    
    [CloudQueueProcessor(4, true)]
    [CloudQueue(true)]
    public class PushNotificationWorker : BaseCloudQueueShoveledMessage<PushNotificationWorker>
    {
        /// <summary>
        /// unique ID of the mobile device
        /// </summary>
        public string DeviceID
        {
            get;
            set;
        }

        /// <summary>
        /// unique ID of the mobile device for VOIP
        /// </summary>
        public string PushKitToken
        {
            get;
            set;
        }

        public string Message
        {
            get;
            set;
        }

        public int[] MentionedUserIDs { get; set; }

        public DevicePlatform Platform
        {
            get;
            set;
        }


        /// <summary>
        /// Who sent the message
        /// </summary>
        public int SenderID
        {
            get;
            set;
        }

        /// <summary>
        /// The name of the user that sent the message (used if the recipient is not a friend)
        /// </summary>
        public string SenderName
        {
            get;
            set;
        }

        /// <summary>
        /// The receiver of the message
        /// </summary>
        public int RecipientID
        {
            get;
            set;
        }

        /// <summary>
        /// The conversation this is associated with (Group or Friend)
        /// </summary>
        public string ConversationID
        {
            get;
            set;
        }

        public DateTime Timestamp
        {
            get;
            set;
        }

        public PushNotificationType Type
        {
            get;
            set;
        }

        public string InviteCode
        {
            get;
            set;
        }

        public long? AccessToken
        {
            get;
            set;
        }

        public CallDetails CallDetails { get; set; }       

        public static void VoiceInvite(int regionID, ClientEndpoint endpoint, int senderID, string senderName, int recipientID, CallDetails callDetails, long? accessToken)
        {
            new PushNotificationWorker(regionID, endpoint)
            {
                SenderID = senderID,
                SenderName =  senderName,
                RecipientID = recipientID,
                InviteCode = callDetails.InviteCode,
                AccessToken = accessToken,
                ConversationID = callDetails.ConversationID,
                Type = PushNotificationType.IncomingCall,
                Timestamp = DateTime.UtcNow,
                CallDetails = callDetails
            }.Enqueue();
        }

        public static void ConversationMessage(int regionID, ClientEndpoint endpoint, int senderID, string senderName, int recipientID, string message, string conversationID, DateTime timestamp, int[] mentionedUserIDs = null)
        {
            new PushNotificationWorker(regionID, endpoint)
            {
                SenderID = senderID,
                RecipientID = recipientID,
                ConversationID = conversationID,
                Message = message,
                Timestamp = timestamp,
                Type = PushNotificationType.ConversationMessage,
                MentionedUserIDs = mentionedUserIDs,
                SenderName = senderName
            }.Enqueue();
        }

        public static void FriendChange(int regionID, ClientEndpoint endpoint, PushNotificationType type, int senderID, int recipientID)
        {
            new PushNotificationWorker(regionID, endpoint)
            {
                SenderID = senderID,
                RecipientID = recipientID,
                Timestamp = DateTime.UtcNow,
                Type = type
            }.Enqueue();
        }

        [JsonConstructor]
        public PushNotificationWorker() { }

        public PushNotificationWorker(int regionID, ClientEndpoint ep)
            : base(regionID)
        {
            Platform = ep.Platform;
            DeviceID = ep.DeviceID;
            PushKitToken = ep.PushKitToken;
        }
    }
}
