﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Newtonsoft.Json;

namespace Curse.Friends.Data.Queues
{
    [CloudQueueProcessor(24, true)]
    [CloudQueue(true)]
    public class RegionalUserChangeResolver : BaseCloudQueueShoveledMessage<RegionalUserChangeResolver>
    {
        [JsonConstructor]
        public RegionalUserChangeResolver() { }

        protected RegionalUserChangeResolver(int destinationRegionID)
            : base(destinationRegionID)
        {
            
        }

        public int UserID
        {
            get;
            set;
        }

        public UserStatistics UserStatistics
        {
            get;
            set;
        }

        public int? FriendID
        {
            get;
            set;
        }

        public static void Create(int userID, bool notifyFriends = true)
        {
            var region = UserRegion.GetByUserID(userID);
            var user = region.GetUser();
            var userStats = user.GetStatistics();
            var myConnectedEndpoints = ClientEndpoint.GetAllConnected(userID);

            // Tell the user that their user info has changed
            try
            {
                var selfNotification = new UserChangeNotification
                {
                    User = user.ToNotification()
                };

                ClientEndpoint.DispatchNotification(myConnectedEndpoints, endpoint => UserChangeTargetedNotifier.Create(selfNotification, endpoint), null);
                
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to queue user change notifications.");
            }


            if (!notifyFriends)
            {
                return;
            }

            // Get all of this user's friends regions (or default to all regions)
            HashSet<int> regionIDs;
            try
            {
                regionIDs = Friendship.GetFriendRegionIDs(region.RegionID, user.UserID);

                if (!regionIDs.Any() && user.FriendCount > 0)
                {
                    Logger.Warn("Unable to determine user's friend regions.", new { UserID = userID, user.FriendCount });
                    return;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to determine user friend region IDs. All regions will be used.");
                regionIDs = new HashSet<int>(QueueConfiguration.Configurations.Select(p => p.RegionIdentifier));
            }

            // For each region, queue up a change resolver
            foreach (var regionID in regionIDs)
            {
                try
                {
                    new RegionalUserChangeResolver(regionID)
                    {
                        UserID = userID,                       
                        UserStatistics = userStats
                    }.Enqueue();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to quueue user change resolver for region: " + region);
                }
            }
        }

        public static void StartProcessor()
        {
            StartProcessor(ProcessMessage);
        }

        static void ProcessMessage(RegionalUserChangeResolver value)
        {

            // Get the sending user's stats
            var userStats = value.UserStatistics ?? UserStatistics.GetByUserID(value.UserID);

            if (userStats == null)
            {
                Logger.Warn("Unable to process user status. User statistics is null.");
                return;
            }
            
                // Get all of the local friends that this user has confirmed
            var recipients = Friendship.GetAllLocal(p => p.OtherUserID, value.UserID)
                                    .Where(p => p.Status == FriendshipStatus.Confirmed)
                                    .ToArray();
            

            // If user has no friends, no need to continue
            if (recipients.Length == 0)
            {
                return;
            }
            
            var watching = string.IsNullOrEmpty(userStats.WatchingChannelID) ? null : ExternalCommunity.GetByTwitchID(userStats.WatchingChannelID);

            var theirConversations = PrivateConversation.MultiGetLocal(recipients.Select(p => new KeyInfo(p.UserID, p.OtherUserID)))
                                                        .ToDictionary(p => p.UserID);

           
            foreach (var theirFriendship in recipients)
            {
                try
                {
                    // Create a notification
                    var notification = theirFriendship.ToNotification(userStats, theirConversations.GetValueOrDefault(theirFriendship.UserID), watching);

                    // Get endpoints for all this users friends
                    var endpoints = ClientEndpoint.GetAllConnected(theirFriendship.UserID);                  

                    // Dispatch a notification for each endpoint
                    foreach (var endpoint in endpoints)
                    {
                        // Dispatch a notification for each endpoint
                        FriendshipChangeNotifier.Create(endpoint, notification);
                    }

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to process user change for friendship.");
                }
            }

            ProcessGroups(value, userStats);

        }

        static void ProcessGroups(RegionalUserChangeResolver value, UserStatistics stats)
        {
            // Get all of the user's group memberships
            var memberships = GroupMember.GetAllByUserID(value.UserID)
                                         .Where(p => p.RegionID == GroupMember.LocalConfigID && p.IsRootGroup)
                                         .ToArray();

            var activeGroups = Group.MultiGetLocal(memberships.Select(p => new KeyInfo(p.GroupID)))
                                    .Where(p => !string.IsNullOrEmpty(p.MachineName) && p.ShouldUpdateMemberPresence) // Only hosted groups that should fanout presence data
                                    .ToArray();

            if (!activeGroups.Any())
            {
                return;
            }

            foreach (var group in activeGroups)
            {               
                try
                {                    
                    GroupChangeCoordinator.UpdateUserPresence(group, new GroupPresenceContract
                    {
                        ConnectionStatus = stats.ConnectionStatus, 
                        UserID = stats.UserID, 
                        GameID = stats.CurrentGameID,
                        DateLastSeen = stats.DateLastSeen,
                        IsActive = true
                    });
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to enqueue GroupChangeCordinator", new { group.GroupID, group.RegionID, group.MachineName });
                }
            }

        }
    }
}
