﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Timers;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.Enums;
using Curse.Logging;
using Timer = System.Timers.Timer;

namespace Curse.Friends.Data
{
    public class GroupHostManager : ServiceHostManager<GroupHost, Group>
    {
        
    }

    public class TwitchHostManager : ServiceHostManager<TwitchHost, ExternalCommunity>
    {
        
    }

    public class GroupPollHostManager : ServiceHostManager<GroupPollHost, GroupPoll>
    {
        
    }

    public class GroupGiveawayHostManager : ServiceHostManager<GroupGiveawayHost, GroupGiveaway>
    {
        
    }

    public class BattleNetHostManager : ServiceHostManager<BattleNetHost, ExternalGuild>
    {
        
    }

    public class TwitchChatHostManager : ServiceHostManager<TwitchHost, ExternalAccount>
    {
        
    }

    public class ServiceHostManager<THost, THostable> where THost:BaseTable<THost>, IServiceHost, new() where THostable:BaseTable<THostable>, IHostable, new()
    {
        private static readonly ConcurrentDictionary<int, THost[]> _healthyHostsByRegion = new ConcurrentDictionary<int, THost[]>();
        private static readonly ConcurrentDictionary<int, HashSet<string>> _healthyHostsNamesByRegion = new ConcurrentDictionary<int, HashSet<string>>();
        private static readonly HostEnvironment _hostEnvironment = HostEnvironment.Unknown; 
        private static int _currentHostIndex = -1;
        private static readonly string _hostTypeName = typeof (THost).Name;
        private static readonly string _hostableTypeName = typeof (THostable).Name;
        private static readonly LogCategory Logger = new LogCategory(string.Format("{0}Manager", _hostTypeName)) {ReleaseLevel = LogLevel.Trace};

        /// <summary>
        /// TServiceManager tracks a list of all available healthy hosts, so a group can be serviced
        /// by free host.
        /// </summary>
        static ServiceHostManager()
        {

#if CONFIG_DEBUG
            _hostEnvironment = HostEnvironment.Debug;
#elif CONFIG_STAGING
            _hostEnvironment = HostEnvironment.Staging;
#elif CONFIG_RELEASE
            _hostEnvironment = HostEnvironment.Release;
#endif

            foreach (var regionID in ConfigurationRegion.AllRegionIDs)
            {
                _healthyHostsByRegion.TryAdd(regionID, new THost[0]);
                _healthyHostsNamesByRegion.TryAdd(regionID, new HashSet<string>());
            }

            UpdateHostList();
            var timer = new Timer();
            timer.Elapsed += TimerOnElapsed;
            timer.Interval = TimeSpan.FromSeconds(15).TotalMilliseconds;
            timer.Start();
        }

        /// <summary>
        /// updates the list of available healthy hosts every 15 seconds
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="elapsedEventArgs"></param>
        private static void TimerOnElapsed(object sender, ElapsedEventArgs elapsedEventArgs)
        {
            UpdateHostList();
        }

        private static void UpdateHostList()
        {
            try
            {
                // Get hosts from all regions
                foreach (var regionID in _healthyHostsByRegion.Keys)
                {
                    try
                    {
                        var allHosts = BaseTable<THost>.GetAll(regionID, p => p.IndexMode, IndexMode.Default);
                        var healthyHosts = allHosts
                                            .Where(p => (p.Status == ServiceHostStatus.Online)
                                                        && (p.Environment == HostEnvironment.Unknown || p.Environment == _hostEnvironment)
                                                        && p.Version == p.CurrentVersion
                                                        && (DateTime.UtcNow - p.DateUpdated <= TimeSpan.FromMinutes(1)))
                                            .ToArray();

                        var healthyHostNames = new HashSet<string>(healthyHosts.Select(p => p.MachineName));

                        _healthyHostsByRegion.AddOrUpdate(regionID, healthyHosts, (i, hosts) => healthyHosts);
                        _healthyHostsNamesByRegion.AddOrUpdate(regionID, healthyHostNames, (i, hosts) => healthyHostNames);

                        if (healthyHosts.Length == 0 && (StorageConfiguration.CurrentMode == ConfigurationMode.Release || regionID == BaseTable<THost>.LocalConfigID))
                        {
                            Logger.Warn(string.Format("No healthy {0} found out of {1} total and {2} with same version for region {3}.",
                                _hostTypeName, allHosts.Length, allHosts.Count(h => h.CurrentVersion == h.Version), regionID));
                        }

                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, string.Format("Failed to get {0} from region: {1}", _hostTypeName, regionID));
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, string.Format("Failed to setup initial {0} list.", _hostTypeName));
            }
        }

        public static THost GetNextHost(int regionID)
        {
            THost[] healthyHosts;

            if (!_healthyHostsByRegion.TryGetValue(regionID, out healthyHosts))
            {
                Logger.Warn("No healthy hosts in region: " + regionID);
                return null;
            }

            if (healthyHosts.Length == 0)
            {
                Logger.Warn("No healthy hosts in region: " + regionID);
                return null;
            }

            var nextIndex = Math.Abs(Interlocked.Increment(ref _currentHostIndex)%healthyHosts.Length);
            return healthyHosts[nextIndex];
        }

        public static bool IsValidHost(int regionID, string name)
        {
            HashSet<string> healthyHostNames;

            if (!_healthyHostsNamesByRegion.TryGetValue(regionID, out healthyHostNames))
            {
                // Assume this is a transient issue
                return true;
            }

            if (healthyHostNames == null || healthyHostNames.Count == 0)
            {
                // Assume this is a transient issue
                return true;
            }

            return healthyHostNames.Contains(name);

        }

        /// <summary>
        /// Checks if the group has a machine name to route and assigns one from the list 
        /// of available hosts, if there is none.
        /// </summary>
        /// <param name="hostable"></param>
        public static void EnsureServiceHost(THostable hostable)
        {
            if (!hostable.IsHostable)
            {
                throw new ArgumentException("The hostable object is not currently hostable: " + hostable.DisplayName);
            }

#if DEBUGLOCAL
            if (string.IsNullOrWhiteSpace(hostable.MachineName) || !hostable.MachineName.Equals(Environment.MachineName))
            {
                hostable.MachineName = Environment.MachineName;
                hostable.Update(p => p.MachineName);
                return;
            }
#endif

#if CONFIG_STAGING
            if (hostable.RegionID != BaseTable<THost>.LocalConfigID)
            {
                Logger.Warn(string.Format("Relocating {0} from a remote region to the local region", _hostableTypeName), new {hostable.DisplayName, hostable.RegionID});
                hostable.RegionID = BaseTable<THost>.LocalConfigID;
                hostable.Update(p => p.RegionID);
            }
#endif
            // We already have a group host
            if (!string.IsNullOrEmpty(hostable.MachineName))
            {                
                // Ensure that the host assigned is valid for the region!
                if (IsValidHost(hostable.RegionID, hostable.MachineName))
                {
                    return;
                }
                
                Logger.Trace("Discovered a hostable that is assigned to an invalid host. We will repair it!", new { hostable.DisplayName, hostable.RegionID, hostable.MachineName });
                
            }
            
            // We need to try to assign a new group host
            var host = GetNextHost(hostable.RegionID);

            if (host == null)
            {
                return;
            }

            var machineName = host.MachineName;            
            
            var attempts = 0;
            while (true)
            {
                var lastAttempt = (++attempts) == 10;

                try
                {
                    // Get the latest data for the specified group
                    var freshHostable = BaseTable<THostable>.Get(hostable.RegionID, hostable.KeyObjects);

                    // We have a group!
                    if (!string.IsNullOrEmpty(freshHostable.MachineName) && IsValidHost(freshHostable.RegionID, freshHostable.MachineName))
                    {
                        hostable.MachineName = freshHostable.MachineName;
                        return;
                    }

                    freshHostable.MachineName = machineName;
                    freshHostable.Update(UpdateMode.Concurrent, p => p.MachineName);
                    hostable.MachineName = freshHostable.MachineName;
                    return;
                }
                catch (AerospikeException ex)
                {
                    if (ex.Message.Contains("Generation error"))
                    {
                        Logger.Debug(ex, string.Format("Soft fail updating {0} host.", _hostableTypeName));    
                    }
                    else
                    {
                        Logger.Error(ex, string.Format("Aerospike error while updating {0} host.", _hostableTypeName));
                    }
                    
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, string.Format("Error updating {0} host.", _hostableTypeName));
                }

                if (lastAttempt)
                {
                    Logger.Error(string.Format("Unable to assign a {0} to a {1}!", _hostTypeName, _hostableTypeName), new {attempts, hostable.KeyObjects, hostable.RegionID});
                    return;
                }

            }
        }
    }
}
