﻿using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.Extensions;
using System;
using System.Linq;

namespace Curse.Friends.Data
{
    public class UserDissolveWorker : BaseCloudQueueWorkerMessage<UserDissolveWorker>
    {
        public int UserID { get; set; }

        public static void Create(int userID)
        {
            new UserDissolveWorker
            {
                UserID = userID,
            }.Enqueue();
        }

        public static void StartProcessor()
        {
            StartProcessor(Process);
        }

        private static void Process(UserDissolveWorker worker)
        {
            var userRegion = UserRegion.GetLocal(worker.UserID);
            if(userRegion == null)
            {
                return;
            }

            var user = userRegion.GetUser();
            if (user == null)
            {
                return;
            }

            // Unmap external accounts
            try
            {
                var mappings = ExternalAccountMapping.GetAllLocal(m => m.UserID, user.UserID).Where(m => !m.IsDeleted).ToArray();
                if (mappings.Any())
                {
                    var accounts = ExternalAccount.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type)));
                    foreach (var account in accounts)
                    {
                        account.DeleteMapping(user.UserID);
                    }
                }
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Failed to unmap user's external accounts during dissolve, step may only be partially complete", worker);
            }

            // Dissolve friendships
            try
            {
                var friendships = Friendship.GetAll(userRegion.RegionID, f => f.UserID, user.UserID);
                var otherRegions = UserRegion.MultiGetLocal(friendships.Select(f => new KeyInfo(f.OtherUserID))).ToDictionary(u => u.UserID);

                var regionalFriendships = friendships.GroupBy(f => f.OtherUserRegionID).ToArray();
                var otherFriendships = regionalFriendships.SelectMany(g => Friendship.MultiGet(g.Key, g.Select(f => new KeyInfo(f.OtherUserID, f.UserID)))).ToDictionary(f => f.UserID);
                var otherUsers = regionalFriendships.SelectMany(g => User.MultiGet(g.Key, g.Select(f => new KeyInfo(f.OtherUserID)))).ToDictionary(u => u.UserID);
                foreach (var friendship in friendships)
                {
                    Friendship theirs;
                    if (!otherFriendships.TryGetValue(friendship.OtherUserID, out theirs))
                    {
                        // Other friendship missing, just delete ours
                        friendship.DurableDelete();
                        continue;
                    }

                    User them;
                    UserRegion theirRegion;
                    if (!otherRegions.TryGetValue(friendship.OtherUserID, out theirRegion) || !otherUsers.TryGetValue(friendship.OtherUserID, out them))
                    {
                        continue;
                    }

                    Friendship.Remove(false, user, friendship, userRegion, them, theirs, theirRegion);
                    new FriendshipRemovedResolver { UserID = friendship.OtherUserID, FriendID = friendship.UserID }.Enqueue();

                    ((IConversationParent)theirs.GetConversation()).ToggleHidden(true);
                }
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Failed to dissolve friendships during dissolve, step may only be partially complete", worker);
            }

            // Dissolve memberships
            try
            {
                var memberships = GroupMember.GetAllLocal(m => m.UserID, user.UserID).Where(m => m.IsRootGroup && !m.IsDeleted).ToArray();
                var groups = Group.MultiGetLocal(memberships.Select(m => new KeyInfo(m.GroupID))).ToDictionary(m => m.GroupID);
                foreach (var membership in memberships)
                {
                    Group group;
                    if (!groups.TryGetValue(membership.GroupID, out group))
                    {
                        membership.SetDeleted();
                        continue;
                    }

                    try
                    {
                        group.Leave(user.UserID, userRegion.RegionID);
                    }
                    catch (GroupPermissionException)
                    {
                        membership.SetDeleted();
                    }
                }
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Failed to leave groups during dissolve, step may only be partially complete", worker);
            }

            // Clear user info
            try
            {
                user.Username = Guid.NewGuid().ToString("N");
                user.DisplayName = user.Username;
                user.TwitchID = string.Empty;
                user.EmailAddress = string.Empty;
                user.EmailVerified = false;
                user.HasSyncedAccount = false;
                user.IsMerged = false;
                user.Update(u => u.Username, u => u.DisplayName, u => u.EmailAddress, u => u.EmailVerified, u => u.TwitchID, u => u.HasSyncedAccount, u => u.IsMerged);

                user.UpdateStatistics(DateTime.UtcNow.ToEpochMilliseconds());
                user.ReindexSearchHints();
            }
            catch(Exception ex)
            {
                Logger.Error(ex, "Failed to change user info during dissolve", worker);
            }
        }
    }
}
