﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Friends.Enums;
using Curse.Logging;
using Nest;

namespace Curse.Friends.Data.Search
{
    public class ExternalCommunitySearchManager : CloudSearchManager<ExternalCommunitySearchModel>
    {
        public static readonly int MaxPageSize = 100;
        private static readonly int _defaultPageSize = 30;

        public static void UpsertCommunity(ExternalCommunity community)
        {
            var searchModel = new ExternalCommunitySearchModel(community);

            var response = GetClient().Update<ExternalCommunitySearchModel>(u => u
                .Id(searchModel.ID)
                .Doc(searchModel)
                .DocAsUpsert());

            if (!response.IsValid)
            {                
                Logger.Warn("Upsert failed", new { community.ExternalID });
            }
        }

        public static void BulkUpsertCommunities(IEnumerable<ExternalCommunity> communities)
        {
            var searchModels = communities.Select(c => new ExternalCommunitySearchModel(c)).ToArray();
            var request = new BulkRequest
            {
                Operations = searchModels.Select(c => (IBulkOperation) new BulkUpdateOperation<ExternalCommunitySearchModel, ExternalCommunitySearchModel>(c.ID)
                {
                    DocAsUpsert = true,
                    Doc = c
                }).ToArray()
            };
            var response = GetClient().Bulk(request);

            if (!response.IsValid)
            {
                Logger.Warn("Upsert failed", new {searchModels, response.ItemsWithErrors});
            }
        }

        public static ExternalCommunitySearchModel[] Search(ExternalCommunitySearch search)
        {
            var queries = new List<QueryContainer>();

            if (!string.IsNullOrWhiteSpace(search.Name))
            {
                queries.Add(new QueryDescriptor<ExternalCommunitySearchModel>().Bool(b => b.Should(
                    s => s.Match(m => m.OnField(f => f.ExternalName).Query(search.Name)),
                    s => s.Match(m => m.OnField(f => f.ExternalDisplayName).Query(search.Name)))));
            }

            if (!string.IsNullOrWhiteSpace(search.Query))
            {
                queries.Add(new QueryDescriptor<ExternalCommunitySearchModel>().Bool(b => b.Should(
                    bs => bs.Term(t => t.ExternalName.Suffix(AutocompleteSuffix), search.Query.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.ExternalName).Query(search.Query)),
                    bs => bs.Term(t => t.ExternalDisplayName.Suffix(AutocompleteSuffix), search.Query.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.ExternalDisplayName).Query(search.Query)),
                    bs => bs.Match(m => m.OnField(f => f.ExternalStatus).Query(search.Query))
                )));
            }

            if (search.IsLive.HasValue)
            {
                queries.Add(new QueryDescriptor<ExternalCommunitySearchModel>().Term(f => f.IsLive, search.IsLive.Value));
            }

            var client = GetClient();


            
            var size = search.PageSize ?? _defaultPageSize;
            size = size < 1 ? 1 : size;
            size = size > MaxPageSize ? MaxPageSize : size;
            var skip = ((search.PageNumber ?? 1) - 1) * size;

            Func<SearchDescriptor<ExternalCommunitySearchModel>, SearchDescriptor<ExternalCommunitySearchModel>> searchDescriptor;

            switch (search.SortType)
            {
                case ExternalCommunitySearchSortType.DateLive:
                    searchDescriptor = s => s
                        .Query(q => q.Bool(b => b
                            .Must(queries.ToArray())))
                        .Size(size)
                        .Skip(skip)
                        .Sort(d => d.OnField(f => f.LiveTimestamp).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case ExternalCommunitySearchSortType.Title:
                    searchDescriptor = s => s
                        .Query(q => q.Bool(b => b
                            .Must(queries.ToArray())))
                        .Size(size)
                        .Skip(skip)
                        .Sort(d=>d.OnField(f=>f.ExternalName.Suffix(SortSuffix)).Order(search.SortAscending?SortOrder.Ascending : SortOrder.Descending));
                    break;
                default:
                    searchDescriptor = s => s
                        .Query(q => q.Bool(b => b
                            .Must(queries.ToArray())))
                        .Size(size)
                        .Skip(skip)
                        .Sort(d => d.OnField(f => f.Quality).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
            }

            var result = client.Search(searchDescriptor);

            if (!result.ConnectionStatus.Success)
            {
                Logger.Warn("Failed to run external community search", search);
            }

            return result.Documents.ToArray();
        }

    }
}
