﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Friends.Enums;
using Curse.Logging;
using Elasticsearch.Net;
using Nest;

namespace Curse.Friends.Data.Search
{
    public class ExternalCommunityMemberManager : CloudSearchManager<ExternalCommunityMemberSearchModel>
    {
        public static readonly int MaxDocumentRequestCount = 500;
        private const string _indexName = "externalcommunitymembers-index";
        private const int _defaultPageSize = 25;

        public static ExternalCommunityMemberSearchModel[] Search(ExternalCommunityMemberSearch search)
        {
            var client = GetClient();
            var res = client.Search(GetSearchRequest(search));
            return res.Documents.ToArray();
        }

        public static bool CountByStatus(string externalCommunityID, AccountType type, GroupRoleTag externalRole, out long actives, out long inactives)
        {
            actives = 0;
            inactives = 0;

            var client = GetClient();
            var res = client.Search<ExternalCommunityMemberSearchModel>(s => s
                .Index(_indexName)
                .SearchType(SearchType.Count)
                .Query(q =>
                    q.Term(t => t.ExternalCommunityID, externalCommunityID) &&
                    q.Term(t => t.ExternalCommunityType, type) &&
                    q.Term(t => t.ExternalCommunityRole, externalRole))
                .Aggregations(a => a.Terms("statuses", ad => ad.Field(f => f.IsDeleted)))
                );

            if (!res.ConnectionStatus.Success)
            {
                return false;
            }

            foreach (var status in res.Aggs.Terms("statuses").Items)
            {
                if (status.KeyAsString == "false")
                {
                    actives = status.DocCount;
                }
                else if (status.KeyAsString == "true")
                {
                    inactives = status.DocCount;
                }
            }

            return true;
        }

        private static Func<SearchDescriptor<ExternalCommunityMemberSearchModel>, SearchDescriptor<ExternalCommunityMemberSearchModel>> GetSearchRequest(ExternalCommunityMemberSearch search)
        {
            var musts = new List<FilterContainer>();
            var queries = new List<QueryContainer>();

            // required fields

            musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Term(m => m.ExternalCommunityType, search.ExternalCommunityType));

            if (!string.IsNullOrWhiteSpace(search.ExternalCommunityID))
            {
                musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Term(m => m.ExternalCommunityID, search.ExternalCommunityID));
            }

            if (!string.IsNullOrWhiteSpace(search.ExternalUserID))
            {
                musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Term(m => m.ExternalUserID, search.ExternalUserID));
            }

            if (search.ExternalCommunityRoles != null && search.ExternalCommunityRoles.Any())
            {
                musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Bool(b => b.Should(s => s.Terms(m => m.ExternalCommunityRole, search.ExternalCommunityRoles))));
            }

            if (search.MinEntryNumber.HasValue || search.MaxEntryNumber.HasValue)
            {
                if (search.MinEntryNumber.HasValue && search.MaxEntryNumber.HasValue)
                {
                    musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Range(
                        r => r.OnField(f => f.EntryNumber).GreaterOrEquals(search.MinEntryNumber.Value).LowerOrEquals(search.MaxEntryNumber.Value)));
                }
                else if (search.MinEntryNumber.HasValue)
                {
                    musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Range(
                        r => r.OnField(f => f.EntryNumber).GreaterOrEquals(search.MinEntryNumber.Value)));
                }
                else
                {
                    musts.Add(new FilterDescriptor<ExternalCommunityMemberSearchModel>().Range(
                        r => r.OnField(f => f.EntryNumber).LowerOrEquals(search.MaxEntryNumber.Value)));
                }
            }

            if (!string.IsNullOrWhiteSpace(search.Query))
            {
                queries.Add(new QueryDescriptor<ExternalCommunityMemberSearchModel>().Bool(b => b.Should(
                    bs => bs.Term(t => t.ExternalUserID, search.Query),
                    bs => bs.Match(m => m.OnField(f => f.ExternalUsername).Query(search.Query)),
                    bs => bs.Term(t => t.ExternalUsername.Suffix(AutocompleteSuffix), search.Query.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.ExternalUserDisplayName).Query(search.Query)),
                    bs => bs.Term(t => t.ExternalUserDisplayName.Suffix(AutocompleteSuffix), search.Query.ToLowerInvariant())
                    )));
            }

            if (!string.IsNullOrWhiteSpace(search.ExternalUsername))
            {
                queries.Add(new QueryDescriptor<ExternalCommunityMemberSearchModel>().Bool(b => b.Should(
                    bs => bs.Match(m => m.OnField(f => f.ExternalUsername).Query(search.ExternalUsername)),
                    bs => bs.Term(t => t.ExternalUsername.Suffix(AutocompleteSuffix), search.ExternalUsername.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.ExternalUserDisplayName).Query(search.ExternalUsername)),
                    bs => bs.Term(t => t.ExternalUserDisplayName.Suffix(AutocompleteSuffix), search.ExternalUsername.ToLowerInvariant())
                    )));
            }

            if (!string.IsNullOrWhiteSpace(search.ExternalCommunityName))
            {
                queries.Add(new QueryDescriptor<ExternalCommunityMemberSearchModel>().Bool(b => b.Should(
                    bs => bs.Match(m => m.OnField(f => f.ExternalCommunityName).Query(search.ExternalCommunityName)),
                    bs => bs.Term(t => t.ExternalCommunityName.Suffix(AutocompleteSuffix), search.ExternalCommunityName.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.ExternalCommunityDisplayName).Query(search.ExternalCommunityName)),
                    bs => bs.Term(t => t.ExternalCommunityDisplayName.Suffix(AutocompleteSuffix), search.ExternalCommunityName.ToLowerInvariant())
                    )));
            }

            var pageSize = search.PageSize ?? _defaultPageSize;
            pageSize = pageSize < 1 ? 1 : pageSize;
            pageSize = pageSize > MaxDocumentRequestCount ? MaxDocumentRequestCount : pageSize;
            var skip = ((search.PageNumber??1) - 1)*pageSize;
            skip = skip < 0 ? 0 : skip;
            return s => s.Index(_indexName)
                .Size(pageSize)
                .Skip(skip)
                .Query(q => q.Bool(b => b.Must(queries.ToArray())))
                .Filter(f => f.Bool(b => b.Must(musts.ToArray())))
                .SortAscending(f => f.FirstRoleDate);
        }

        public static void Index(ExternalCommunityMemberSearchModel member)
        {
            var client = GetClient();

            var res = client.Index(member, i => i
                .Index(_indexName));

            if (!res.ConnectionStatus.Success)
            {
                Logger.Trace("Failed to index an external community member", member);
            }
        }

        public static IEnumerable<IEnumerable<ExternalCommunityMemberSearchModel>> Scroll(ExternalCommunityMemberSearch search)
        {
            var client = GetClient();
            var result = client.Search<ExternalCommunityMemberSearchModel>(s => GetSearchRequest(search)(s).Scroll("10s"));
            while (result.Documents.Any())
            {
                yield return result.Documents;
                result = client.Scroll<ExternalCommunityMemberSearchModel>("10s", result.ScrollId);
            }
        }

        public static void BulkUpdateUserID(IEnumerable<ExternalCommunityMemberSearchModel> members, int userID, bool added)
        {
            var client = GetClient();

            var operations = new List<IBulkOperation>(members.Select(m => new BulkUpdateDescriptor<ExternalCommunityMemberSearchModel, ExternalCommunityMemberSearchModel>()
                .Id(m.ModelID)
                .Upsert(m)
                .Script(added ? "ctx._source.userIds+=userId;" : "ctx._source.userIds-=userId;")
                .Params(p => p.Add("userId", userID))
                .RetriesOnConflict(3)));

            var request = new BulkRequest
            {
                Index = _indexName,
                Operations = operations,
                Refresh = true
            };
            var res = client.Bulk(request);
            if (!res.ConnectionStatus.Success)
            {
                throw new InvalidOperationException();
            }
        }

        private static readonly string _updateScript = string.Join("",
            "ctx._source.currentRoleDate=currentRoleDate;",
            "ctx._source.isDeleted=isDeleted;",
            "ctx._source.userIds=userIds;");

        private static FluentDictionary<string, object> GetUpdateParameters(FluentDictionary<string, object> parameters, ExternalCommunityMemberSearchModel member)
        {
            return parameters
                .Add("currentRoleDate", member.CurrentRoleDate)
                .Add("isDeleted", member.IsDeleted)
                .Add("userIds", member.UserIDs);
        } 

        public static void BulkUpsert(IEnumerable<ExternalCommunityMemberSearchModel> members)
        {
            var client = GetClient();

            var request = new BulkRequest
            {
                Index = _indexName,
                Operations = members.Select(m => (IBulkOperation)new BulkUpdateDescriptor<ExternalCommunityMemberSearchModel, ExternalCommunityMemberSearchModel>()
                .Id(m.ModelID)
                .Upsert(m)
                .Script(_updateScript)
                .Params(p=>GetUpdateParameters(p, m))
                .RetriesOnConflict(3)).ToArray()
            };

            var res = client.Bulk(request);

            if (!res.ConnectionStatus.Success)
            {
                throw new InvalidOperationException();
            }
        }

        public static void UpsertMembership(ExternalCommunityMemberSearchModel member)
        {
            var client = GetClient();

            var res = client.Update<ExternalCommunityMemberSearchModel>(u => u
                .Index(_indexName)
                .Id(member.ModelID)
                .Upsert(member)
                .Script(_updateScript)
                .Params(p=>GetUpdateParameters(p,member))
                .RetryOnConflict(3));

            if (!res.ConnectionStatus.Success)
            {
                throw new InvalidOperationException();
            }
        }
    }
}
