﻿using System.Linq;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Nest;

namespace Curse.Friends.Data.Search
{
    [CloudSearchModel(AutoCreateIndex = false, UseDefaultIndex = false, IndexTypeName = "externalcommunitymembers")]
    [ElasticType(Name = "externalCommunityMember", IdProperty = "modelId")]
    public class ExternalCommunityMemberSearchModel
    {

        public ExternalCommunityMemberSearchModel()
        {

        }

        private ExternalCommunityMemberSearchModel(string externalUserID, string externalCommunityID, AccountType communityType, GroupRoleTag roleTag)
        {
            ModelID = CreateModelID(externalUserID, externalCommunityID, communityType, roleTag);
            ExternalUserID = externalUserID;
            ExternalCommunityID = externalCommunityID;
            ExternalCommunityType = communityType;
            ExternalCommunityRole = roleTag;
        }

        public ExternalCommunityMemberSearchModel(ExternalCommunityMembership membership, int[] userIDs)
            : this(membership.ExternalUserID, membership.ExternalCommunityID, membership.Type, membership.RoleTag)
        {
            ExternalUsername = membership.ExternalUsername;
            ExternalUserDisplayName = membership.ExternalUserDisplayName;

            ExternalCommunityName = membership.ExternalCommunityName;
            ExternalCommunityDisplayName = membership.ExternalCommunityDisplayName;
            
            CurrentRoleDate = membership.CurrentRoleDate.ToEpochMilliseconds();
            FirstRoleDate = membership.FirstRoleDate.ToEpochMilliseconds();
            
            IsDeleted = membership.Status == ExternalCommunityMembershipStatus.Deleted;
            EntryNumber = membership.EntryNumber;
            UserIDs = userIDs;
        }

        public static string CreateModelID(string externalUserID, string externalCommunityID, AccountType communityType, GroupRoleTag externalCommunityRole)
        {
            return string.Format("{0}:{1}:{2}:{3}", externalCommunityID, communityType, externalUserID, externalCommunityRole);
        }

        [ElasticProperty(Name = "modelId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string ModelID { get; set; }

        [ElasticProperty(Name = "externalUserId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string ExternalUserID { get; set; }

        [ElasticProperty(Name = "externalCommunityId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string ExternalCommunityID { get; set; }

        [ElasticProperty(Name = "externalCommunityType", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public AccountType ExternalCommunityType { get; set; }

        [ElasticProperty(Name = "externalCommunityRole", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public GroupRoleTag ExternalCommunityRole { get; set; }

        [AnalyzedStringProperty("externalUsername")]
        public string ExternalUsername { get; set; }

        [AnalyzedStringProperty("externalDisplayName")]
        public string ExternalUserDisplayName { get; set; }

        [AnalyzedStringProperty("externalCommunityName")]
        public string ExternalCommunityName { get; set; }

        [AnalyzedStringProperty("externalCommunityDisplayName")]
        public string ExternalCommunityDisplayName { get; set; }

        [ElasticProperty(Name = "userIds", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int[] UserIDs { get; set; }

        [ElasticProperty(Name = "currentRoleDate", Type = FieldType.Long, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public long CurrentRoleDate { get; set; }

        [ElasticProperty(Name = "firstRoleDate", Type = FieldType.Long, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public long FirstRoleDate { get; set; }

        [ElasticProperty(Name = "isDeleted", Type = FieldType.Boolean, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public bool IsDeleted { get; set; }

        [ElasticProperty(Name = "entryNumber", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int EntryNumber { get; set; }

        public ExternalCommunityMemberContract ToNotification(GroupMember member, UserStatistics statistics)
        {
            var contract = new ExternalCommunityMemberContract
            {
                ExternalCommunityRole = ExternalCommunityRole,
                ExternalCommunityName = ExternalCommunityName,
                ExternalCommunityID = ExternalCommunityID,
                ExternalCommunityType = ExternalCommunityType,
                ExternalUserID = ExternalUserID,
                ExternalUsername = ExternalUsername,
                ExternalUserDisplayName = ExternalUserDisplayName,
                ExternalRoleCurrentTimestamp = CurrentRoleDate,
                ExternalRoleFirstTimestamp = FirstRoleDate,
                ExternalRoleIsActive = !IsDeleted,
                ExternalRoleEntryNumber = EntryNumber
            };

            if (member != null)
            {
                contract.GroupMember = statistics != null ? member.ToNotification(statistics) : null;
                contract.GroupID = member.GroupID;
                contract.Username = member.GetTitleName();
                contract.UserID = member.UserID;
                contract.GroupRoles = member.Roles.ToArray();
            }

            return contract;
        }
    }
}
