﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Friends.Enums;
using Nest;

namespace Curse.Friends.Data.Search.FriendSearch
{
    public class CharacterFriendSearchManager : CloudSearchManager<CharacterFriendSearchModel>
    {
        public static bool TrySearch(string query, FriendHint[] searchHints, out CloudSearchResult<CharacterFriendSearchModel>[] results, out int searchTime)
        {
            // Remove any invalid characters
            var queryString = query.Replace(" ", string.Empty).Replace("_", string.Empty);

            // Get the current user's character search hints
            var characterHints = searchHints.Where(p => p.Type == FriendHintType.Game).ToArray();
            var distinctRegions = characterHints.Select(p => p.Region).Distinct().ToArray();
            var distinctServers = characterHints.Select(p => p.Server).Distinct().ToArray();

            var shouldFilters = new List<Func<QueryDescriptor<CharacterFriendSearchModel>, QueryContainer>>();

            foreach (var region in distinctRegions)
            {
                shouldFilters.Add(qd => qd.Term(t => t.ServerRegion, region, 2.0));
            }

            shouldFilters.Add(qd => qd.Terms(t => t.ServerName, distinctServers));

            var characterClient = GetClient();

            var characterResults = characterClient.Search<CharacterFriendSearchModel>(s => s
                .Query(q => q
                    .Bool(b => b
                        .Must(
                            qd => qd.Term(t => t.SearchTerm.Suffix(CloudSearchConstants.AutocompleteSuffix), queryString)
                        )
                        .Should(shouldFilters.ToArray()).MinimumShouldMatch(0).Boost(2.0)
                        .Should(sh=>sh.Term(t=>t.SearchTerm.Suffix(CloudSearchConstants.SortSuffix), query)).MinimumShouldMatch(0).Boost(1.5)
                    )
                )
                .Take(10)
                );

            if (!characterResults.IsValid)
            {
                results = null;
                searchTime = characterResults.ElapsedMilliseconds;
                return false;
            }

            searchTime = characterResults.ElapsedMilliseconds;
            results = CloudSearchResult<CharacterFriendSearchModel>.CreateFromQuery(characterResults).ToArray();
            return true;
        }
    }

    public class EmailFriendSearchManager : CloudSearchManager<EmailFriendSearchModel>
    {
        public static bool TrySearch(string query, out CloudSearchResult<EmailFriendSearchModel>[] results)
        {
            var emailSearchClient = GetClient();
            var emailResults = emailSearchClient.Search<EmailFriendSearchModel>(s => s.Query(q => q.Match(m => m.OnField(p => p.EmailAddress).Query(query))).Take(1));

            if (!emailResults.IsValid)
            {
                results = null;
                return false;
            }

            results = emailResults.Total > 0 ? CloudSearchResult<EmailFriendSearchModel>.CreateFromQuery(emailResults) : new CloudSearchResult<EmailFriendSearchModel>[0];
            return true;
        }
    }

    public class PlatformFriendSearchManager : CloudSearchManager<PlatformFriendSearchModel>
    {
        public static bool TrySearch(string query, out CloudSearchResult<PlatformFriendSearchModel>[] results, out int searchTime)
        {
            var platformSearchClient = GetClient();

            var platformResults = platformSearchClient.Search<PlatformFriendSearchModel>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.SearchTerm.Suffix(CloudSearchConstants.LowercaseSuffix), query)))
                && q.Bool(b => b.Must(qd => qd.Term(t => t.Visibility, FriendHintVisibility.VisibleToEveryone)))));
            if (!platformResults.IsValid)
            {
                results = null;
                searchTime = platformResults.ElapsedMilliseconds;
                return false;
            }

            searchTime = platformResults.ElapsedMilliseconds;
            results = CloudSearchResult<PlatformFriendSearchModel>.CreateFromQuery(platformResults).ToArray();
            return true;
        }
    }

    public class UsernameFriendSearchManager : CloudSearchManager<UsernameFriendSearchModel>
    {
        public static bool TrySearch(string query, out CloudSearchResult<UsernameFriendSearchModel>[] results, out int searchTime)
        {
            var usernameSearchClient = GetClient();
            var userResults = usernameSearchClient.Search<UsernameFriendSearchModel>(s =>
                s.Query(q =>
                    q.Term(t =>
                        t.Username.Suffix(CloudSearchConstants.AutocompleteSuffix), query)).Take(5));
            if (!userResults.IsValid)
            {
                results = null;
                searchTime = userResults.ElapsedMilliseconds;
                return false;
            }

            results = CloudSearchResult<UsernameFriendSearchModel>.CreateFromQuery(userResults);
            searchTime = userResults.ElapsedMilliseconds;
            return true;
        }
    }
}
