﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Friends.Enums;
using Nest;

namespace Curse.Friends.Data.Search
{
    public class GroupBannedUserSearchManager : CloudSearchManager<GroupBannedUserSearchModel>
    {
        public static readonly int MaxPageSize = 100;

        private static readonly string _indexName = "bannedusers-index";

        public static GroupBannedUserSearchModel[] Search(GroupBannedUserSearch search)
        {
            var size = search.PageSize ?? 10;
            var skip = ((search.PageNumber ?? 1) - 1)*size;

            var queries = new List<QueryContainer>();
            var filters = new List<FilterContainer>
            {
                new FilterDescriptor<GroupBannedUserSearchModel>().Term(f => f.IsDeleted, false),
                new FilterDescriptor<GroupBannedUserSearchModel>().Term(f => f.GroupID, search.GroupID),
            };

            if (!string.IsNullOrWhiteSpace(search.Query))
            {
                queries.Add(new QueryDescriptor<GroupBannedUserSearchModel>().Bool(b => b.Should(
                    bs => bs.Match(m => m.OnField(f => f.Username).Query(search.Query)),
                    bs => bs.Match(m => m.OnField(f => f.Username.Suffix(AutocompleteSuffix)).Query(search.Query)),
                    bs => bs.Match(m => m.OnField(f => f.RequestorUsername).Query(search.Query)),
                    bs => bs.Match(m => m.OnField(f => f.RequestorUsername.Suffix(AutocompleteSuffix)).Query(search.Query)),
                    bs => bs.Match(m => m.OnField(f => f.Reason).Query(search.Query))
                    )));
            }

            if (!string.IsNullOrWhiteSpace(search.Username))
            {
                queries.Add(new QueryDescriptor<GroupBannedUserSearchModel>().Bool(b => b.Should(
                    bs => bs.Match(m => m.OnField(f => f.Username).Query(search.Username)),
                    bs => bs.Match(m => m.OnField(f => f.Username.Suffix(AutocompleteSuffix)).Query(search.Username))
                    )));
            }

            if (!string.IsNullOrWhiteSpace(search.RequestorUsername))
            {
                queries.Add(new QueryDescriptor<GroupBannedUserSearchModel>().Bool(b => b.Should(
                    bs => bs.Match(m => m.OnField(f => f.RequestorUsername).Query(search.Username)),
                    bs => bs.Match(m => m.OnField(f => f.RequestorUsername.Suffix(AutocompleteSuffix)).Query(search.Query))
                    )));
            }

            if (!string.IsNullOrWhiteSpace(search.Reason))
            {
                queries.Add(new QueryDescriptor<GroupBannedUserSearchModel>().Bool(b => b.Should(
                    bs => bs.Match(m => m.OnField(f => f.Reason).Query(search.Username))
                    )));
            }

            var searchDescriptor = new SearchDescriptor<GroupBannedUserSearchModel>()
                .Index(_indexName)
                .Routing(search.GroupID.ToString())
                .Query(q => q.Bool(b => b.Must(queries.ToArray())))
                .Filter(f => f.Bool(b => b.Must(filters.ToArray())))
                .Size(size)
                .Skip(skip);

            var sortAscending = search.SortAscending ?? true;
            switch (search.SortType)
            {
                case GroupBannedUserSortType.Date:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.StatusTimestamp).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case GroupBannedUserSortType.Reason:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.Reason.Suffix(SortSuffix)).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case GroupBannedUserSortType.Requestor:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.RequestorUsername.Suffix(SortSuffix)).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case GroupBannedUserSortType.Username:
                default:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.Username.Suffix(SortSuffix)).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
            }

            var client = GetClient();
            var resp = client.Search<GroupBannedUserSearchModel>(s=>searchDescriptor);

            return resp.Documents.ToArray();
        }

        public static GroupBannedUserSearchModel GetByID(Guid groupID, int userID)
        {
            var client = GetClient();
            var resp = client.Get<GroupBannedUserSearchModel>(g => g
                .Id(GroupBannedUserSearchModel.GetModelID(groupID, userID))
                .Index(_indexName)
                .Routing(groupID.ToString()));

            return resp.Source;
        }

        public static void Index(GroupBannedUserSearchModel model)
        {
            var client = GetClient();
            var resp = client.Index(model, i => i
                .Index(_indexName)
                .Routing(model.GroupID.ToString())
                .Id(model.ModelID));

            if (!resp.ConnectionStatus.Success)
            {
                throw new Exception("Failed to index banned user model: " + model.ModelID);
            }
        }
    }
}
