﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Nest;

namespace Curse.Friends.Data.Search
{
    [CloudSearchModel(AutoCreateIndex = false, UseDefaultIndex = false, IndexTypeName = "groupevents", UseAlias = true)]
    [ElasticType(Name = "groupEvent", IdProperty = "eventId")]
    public class GroupEvent
    {
        public GroupEvent()
        {

        }

        private GroupEvent(Guid groupID, GroupEventCategory category, GroupEventType type, int initiatingUserID, string initiatingUsername)
        {
            EventID = Guid.NewGuid();
            Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
            GroupID = groupID;
            EventCategory = category;
            EventType = type;
            InitiatingMemberUserID = initiatingUserID;
            InitiatingMemberUsername = initiatingUsername;
        }

        [ElasticProperty(Name = "eventId", Type = FieldType.String, Index = FieldIndexOption.NotAnalyzed)]
        public Guid EventID { get; set; }

        [ElasticProperty(Name = "groupId", Type = FieldType.String, Index = FieldIndexOption.NotAnalyzed)]
        public Guid GroupID { get; set; }

        [ElasticProperty(Name = "eventCategory", Type = FieldType.Integer, Index = FieldIndexOption.NotAnalyzed)]
        public GroupEventCategory EventCategory { get; set; }

        [ElasticProperty(Name = "eventType", Type = FieldType.Integer, Index = FieldIndexOption.NotAnalyzed)]
        public GroupEventType EventType { get; set; }

        [ElasticProperty(Name = "timestamp", Type = FieldType.Long, Index = FieldIndexOption.NotAnalyzed)]
        public long Timestamp { get; set; }

        [ElasticProperty(Name = "initiatingMemberUserId", Type = FieldType.Integer, Index = FieldIndexOption.NotAnalyzed)]
        public int InitiatingMemberUserID { get; set; }

        [AnalyzedStringProperty("initiatingMemberUsername")]
        public string InitiatingMemberUsername { get; set; }

        #region Root Group

        [ElasticProperty(Name = "rootGroupChangeFlags", Type = FieldType.Integer)]
        public GroupEventRootChangeFlags? RootGroupChangeFlags { get; set; }

        [ElasticProperty(Name = "previousRootGroupDetails", Type = FieldType.Object)]
        public GroupEventRootGroup PreviousRootGroupDetails { get; set; }

        [ElasticProperty(Name = "rootGroupDetails", Type = FieldType.Object)]
        public GroupEventRootGroup RootGroupDetails { get; set; }

        public static GroupEvent CreateRootGroup(Group group, NewGroupMember creator)
        {
            return new GroupEvent(group.GroupID, GroupEventCategory.Group, GroupEventType.GroupCreated, creator.UserID, creator.Username)
            {
                RootGroupDetails = new GroupEventRootGroup
                {
                    IsPublic = group.IsPublic,
                    Title = group.Title,
                    AfkTimerMinutes = group.AfkTimerMinutes,
                    VoiceRegion = group.VoiceRegionID
                },
            };
        }

        #endregion

        #region Channels

        [ElasticProperty(Name = "channelChangeFlags", Type = FieldType.Integer)]
        public GroupEventChannelChangeFlags? ChannelChangeFlags { get; set; }

        [ElasticProperty(Name = "previousChannelDetails", Type = FieldType.Object)]
        public GroupEventChannel PreviousChannelDetails { get; set; }

        [ElasticProperty(Name = "channelDetails", Type = FieldType.Object)]
        public GroupEventChannel ChannelDetails { get; set; }

        public static GroupEvent CreateChannel(Group parent, Group channel, GroupMember requestor, int[] accessRoles)
        {
            return new GroupEvent(channel.RootGroupID, GroupEventCategory.Channel, GroupEventType.ChannelCreated, requestor.UserID, requestor.GetTitleName())
            {
                ChannelDetails = new GroupEventChannel
                {
                    Type = channel.Type,
                    Title = channel.Title,
                    ChannelID = channel.GroupID,
                    IsPublic = channel.IsPublic,
                    AccessRoles = accessRoles,
                    Mode = channel.Mode
                }
            };
        }

        public static GroupEvent DeleteChannel(Group parent, Group channel, GroupMember requestor)
        {
            return new GroupEvent(channel.RootGroupID, GroupEventCategory.Channel, GroupEventType.ChannelRemoved, requestor.UserID, requestor.GetTitleName())
            {
                PreviousChannelDetails = new GroupEventChannel
                {
                    Type = channel.Type,
                    Title = channel.Title,
                    ChannelID = channel.GroupID,
                    Mode = channel.Mode
                }
            };
        }


        #endregion

        #region Roles

        [ElasticProperty(Name = "roleChangeFlags", Type = FieldType.Integer)]
        public GroupEventRoleChangeFlags? RoleChangeFlags { get; set; }

        [ElasticProperty(Name = "previousRoleDetails", Type = FieldType.Object)]
        public GroupEventRole PreviousRoleDetails { get; set; }

        [ElasticProperty(Name = "roleDetails", Type = FieldType.Object)]
        public GroupEventRole RoleDetails { get; set; }


        #endregion

        #region Members

        [ElasticProperty(Name = "memberDetails", Type = FieldType.Object)]
        public GroupEventMember MemberDetails { get; set; }

        public static GroupEvent[] ChangeMembership(Group group, GroupMember requestor, IEnumerable<NewGroupMember> affected, GroupEventType changeType)
        {
            return affected.Select(user =>
                new GroupEvent(group.GroupID, GroupEventCategory.User, changeType, requestor.UserID, requestor.GetTitleName())
                {
                    MemberDetails = new GroupEventMember
                    {
                        UserID = user.UserID,
                        Username = user.Username,
                    }
                }).ToArray();
        }

        public static GroupEvent[] ChangeMembership(Group group, NewGroupMember requestor, IEnumerable<NewGroupMember> affected, GroupEventType changeType)
        {
            return affected.Select(user =>
                new GroupEvent(group.GroupID, GroupEventCategory.User, changeType, requestor.UserID, requestor.Username)
                {
                    MemberDetails = new GroupEventMember
                    {
                        UserID = user.UserID,
                        Username = user.Username,
                    }
                }).ToArray();
        }

        public static GroupEvent[] ChangeMembership(Group group, GroupMember requestor, IEnumerable<GroupMember> affected, GroupEventType changeType)
        {
            return affected.Select(user =>
                new GroupEvent(group.GroupID, GroupEventCategory.User, changeType, requestor.UserID, requestor.GetTitleName())
                {
                    MemberDetails = new GroupEventMember
                    {
                        UserID = user.UserID,
                        Username = user.GetTitleName(),
                    }
                }).ToArray();
        }

        public static GroupEvent ChangeUserRole(Group root, GroupMember requestor, GroupMember otherMember, GroupRole role, GroupEventType changeType)
        {
            return new GroupEvent(root.GroupID, GroupEventCategory.User, changeType, requestor.UserID, requestor.GetTitleName())
            {
                MemberDetails = new GroupEventMember
                {
                    UserID = otherMember.UserID,
                    Username = otherMember.GetTitleName(),
                    RoleID = role.RoleID,
                    RoleName = role.Name,
                }
            };
        }


        public static GroupEvent ChangeMemberNickname(Group group, int userID, string formerUsername, string newUsername)
        {
            return new GroupEvent(group.GroupID, GroupEventCategory.User, GroupEventType.UserNickname, userID, formerUsername)
            {
                MemberDetails = new GroupEventMember
                {
                    Username = newUsername,
                }
            };

        }

        #endregion

        #region Giveaways

        [ElasticProperty(Name = "giveawayDetails", Type = FieldType.Object)]
        public GroupEventGiveaway GiveawayDetails { get; set; }

        public static GroupEvent FromGiveaway(GroupMember requestor, GroupGiveaway giveaway, GroupEventType eventType, GroupMember winner = null, GroupGiveawayRollStatus? rollStatus = null, GroupRole winnerBestRole = null)
        {
            var groupEvent = new GroupEvent(giveaway.GroupID, GroupEventCategory.Giveaway, eventType, requestor.UserID, requestor.GetTitleName())
            {
                GiveawayDetails = new GroupEventGiveaway
                {
                    Entries = giveaway.TotalEntries,
                    RollNumber = giveaway.CurrentRoll,
                    RollStatus = rollStatus,
                    GiveawayID = giveaway.GiveawayID,
                    Title = giveaway.Title,

                }
            };

            if (winner != null)
            {
                groupEvent.GiveawayDetails.WinnerUserID = winner.UserID;
                groupEvent.GiveawayDetails.WinnerUsername = winner.GetTitleName();
                groupEvent.GiveawayDetails.WinnerBestRoleID = winner.BestRole;
            }

            if (winnerBestRole != null)
            {
                groupEvent.GiveawayDetails.WinnerBestRoleName = winnerBestRole.Name;
            }

            return groupEvent;
        }



        #endregion

        #region Polls

        [ElasticProperty(Name = "pollDetails", Type = FieldType.Object)]
        public GroupEventPoll PollDetails { get; set; }

        public static GroupEvent FromPoll(GroupMember requestor, GroupPoll poll, GroupEventType eventType)
        {
            string[] winningOptions;
            int totalVotes;
            int optionsCount;

            if (poll.OptionIDs != null)
            {
                var opts = poll.GetOptions().ToDictionary(o => o.OptionID);
                totalVotes = opts.Values.Aggregate(0, (total, current) => total + current.VoteCount);
                optionsCount = poll.OptionIDs.Count;
                winningOptions = opts.GroupBy(o => o.Value.VoteCount).OrderByDescending(g => g.Key).First().Select(o => o.Value.OptionText).ToArray();
            }
            else
            {
                // Legacy
                totalVotes = poll.Votes.Values.Sum();
                optionsCount = poll.Options.Count;
                winningOptions = poll.Votes.GroupBy(v => v.Value).OrderByDescending(v => v.Key).First().Select(v => poll.Options[v.Key]).ToArray();
            }

            return new GroupEvent(poll.GroupID, GroupEventCategory.Poll, eventType, requestor.UserID, requestor.GetTitleName())
            {
                PollDetails = new GroupEventPoll
                {
                    RequiredRoles = poll.RequiredRoles.ToArray(),
                    Title = poll.Title,
                    PollID = poll.PollID,
                    TotalVotes = totalVotes,
                    DurationMinutes = poll.EndDate == DateTime.MaxValue ? 0 : (int)Math.Ceiling((poll.EndDate - poll.StartDate).TotalMinutes),
                    OptionsCount = optionsCount,
                    WinningOptions = winningOptions
                }
            };
        }

        #endregion

        #region Linked Communities

        [ElasticProperty(Name = "linkedCommunityChangeFlags", Type = FieldType.Integer)]
        public GroupEventCommunityChangeFlags? LinkedCommunityChangeFlags { get; set; }

        [ElasticProperty(Name = "previousLinkedCommunityDetails", Type = FieldType.Object)]
        public GroupEventLinkedCommunity PreviousLinkedCommunityDetails { get; set; }

        [ElasticProperty(Name = "linkedCommunityDetails", Type = FieldType.Object)]
        public GroupEventLinkedCommunity LinkedCommunityDetails { get; set; }

        public static GroupEvent LinkExternalCommunity(GroupMember requestor, ExternalCommunityMapping mapping)
        {
            return new GroupEvent(mapping.GroupID, GroupEventCategory.CommunityLink, GroupEventType.CommunityLinked,requestor.UserID,requestor.Username)
            {
                LinkedCommunityDetails = new GroupEventLinkedCommunity
                {
                    ExternalID = mapping.ExternalID,
                    ExternalName = mapping.ExternalDisplayName??mapping.ExternalName,
                    GracePeriodAction = mapping.GracePeriodExpireAction,
                    GracePeriodDays = mapping.GracePeriodDays,
                    SyncEmoticons = mapping.SyncEmotes,
                }
            };
        }

        public static GroupEvent UnlinkExternalCommunity(GroupMember requestor, ExternalCommunityMapping mapping)
        {
            return new GroupEvent(mapping.GroupID, GroupEventCategory.CommunityLink, GroupEventType.CommunityUnlinked, requestor.UserID, requestor.Username)
            {
                PreviousLinkedCommunityDetails = new GroupEventLinkedCommunity
                {
                    ExternalID = mapping.ExternalID,
                    ExternalName = mapping.ExternalDisplayName ?? mapping.ExternalName,
                    GracePeriodAction = mapping.GracePeriodExpireAction,
                    GracePeriodDays = mapping.GracePeriodDays,
                    SyncEmoticons = mapping.SyncEmotes,
                }
            };
        }

        #endregion

        #region Linked Guilds

        [ElasticProperty(Name = "previousLinkedGuildDetails", Type = FieldType.Object)]
        public GroupEventLinkedGuild PreviousLinkedGuildDetails { get; set; }

        [ElasticProperty(Name = "linkedGuildDetails", Type = FieldType.Object)]
        public GroupEventLinkedGuild LinkedGuildDetails { get; set; }

        public static GroupEvent LinkExternalGuild(GroupMember requestor, ExternalGuildMapping mapping)
        {
            return new GroupEvent(mapping.GroupID, GroupEventCategory.GuildLink, GroupEventType.GuildLinked, requestor.UserID, requestor.Username)
            {
                LinkedGuildDetails = new GroupEventLinkedGuild
                {
                    Type = mapping.Type,
                    GameRegion = mapping.GameRegion,
                    GameServer = mapping.GameServer,
                    Name = mapping.Name,
                }
            };
        }

        public static GroupEvent UnlinkExternalGuild(GroupMember requestor, ExternalGuildMapping mapping)
        {
            return new GroupEvent(mapping.GroupID, GroupEventCategory.GuildLink, GroupEventType.GuildUnlinked, requestor.UserID, requestor.Username)
            {
                PreviousLinkedGuildDetails = new GroupEventLinkedGuild
                {
                    Type = mapping.Type,
                    GameRegion = mapping.GameRegion,
                    GameServer = mapping.GameServer,
                    Name = mapping.Name,
                }
            };
        }

        #endregion

        public GroupEventContract ToContract()
        {
            return new GroupEventContract
            {
                Type = EventType,
                RootGroupID = GroupID,
                Category = EventCategory,
                Timestamp = Timestamp,
                InitiatingUsername = InitiatingMemberUsername == "Curse" ? "Twitch" : InitiatingMemberUsername,
                InitiatingUserID = InitiatingMemberUserID,

                RootChangeFlags = RootGroupChangeFlags ?? GroupEventRootChangeFlags.None,
                PreviousRootGroupDetails = PreviousRootGroupDetails == null ? null : PreviousRootGroupDetails.ToContract(),
                RootGroupDetails = RootGroupDetails == null ? null : RootGroupDetails.ToContract(),

                ChannelChangeFlags = ChannelChangeFlags ?? GroupEventChannelChangeFlags.None,
                PreviousChannelDetails = PreviousChannelDetails == null ? null : PreviousChannelDetails.ToContract(),
                ChannelDetails = ChannelDetails == null ? null : ChannelDetails.ToContract(),

                RoleChangeFlags = RoleChangeFlags ?? GroupEventRoleChangeFlags.None,
                PreviousRoleDetails = PreviousRoleDetails == null ? null : PreviousRoleDetails.ToContract(),
                RoleDetails = RoleDetails == null ? null : RoleDetails.ToContract(),

                LinkedCommunityChangeFlags = LinkedCommunityChangeFlags ?? GroupEventCommunityChangeFlags.None,
                PreviousLinkedCommunityDetails = PreviousLinkedCommunityDetails == null ? null : PreviousLinkedCommunityDetails.ToContract(),
                LinkedCommunityDetails = LinkedCommunityDetails == null ? null : LinkedCommunityDetails.ToContract(),

                MemberDetails = MemberDetails == null ? null : MemberDetails.ToContract(),
                GiveawayDetails = GiveawayDetails == null ? null : GiveawayDetails.ToContract(),
                PollDetails = PollDetails == null ? null : PollDetails.ToContract(),
            };
        }
    }
}
