﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Logging;
using Nest;

namespace Curse.Friends.Data.Search
{
    public class GroupEventManager : CloudSearchManager<GroupEvent>
    {
        public static GroupEvent[] GetByGroup(Guid groupID, int? pageSize, int? pageNumber)
        {
            var client = GetClient();

            var size = pageSize ?? 25;
            size = size < 1 ? 1 : size;
            size = size > 50 ? 50 : size;
            var skip = ((pageNumber ?? 1) - 1)*size;
            skip = skip < 0 ? 0 : skip;

            var resp = client.Search<GroupEvent>(s => s.Filter(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.GroupID, groupID))))
                .Indices(IndexWildcard)
                .Routing(groupID.ToString())
                .Size(size)
                .Skip(skip)
                .SortDescending(p => p.Timestamp));

            if (!resp.ConnectionStatus.Success)
            {
                Logger.Warn("Unable to retrieve group events: " + resp.ConnectionStatus.HttpStatusCode, new {groupID});
                return new GroupEvent[0];
            }

            var requestParams = Encoding.Default.GetString(resp.RequestInformation.Request);
            Logger.Trace("Request Url: " + resp.RequestInformation.RequestUrl, new {Params = requestParams});
            return resp.Documents.ToArray();
        }

        public static GroupEvent[] Search(GroupEventSearch search)
        {
            var filters = new List<FilterContainer>
            {
                new FilterDescriptor<GroupEvent>().Term(f => f.GroupID, search.GroupID)
            };

            if (search.IncludedEventCategories != null && search.IncludedEventCategories.Length>0)
            {
                filters.Add(new FilterDescriptor<GroupEvent>().Terms(f => f.EventCategory, search.IncludedEventCategories));
            }

            if (search.ExcludedEventCategories != null && search.ExcludedEventCategories.Length > 0)
            {
                filters.Add(new FilterDescriptor<GroupEvent>().Bool(b => b.MustNot(bm => bm.Terms(f => f.EventCategory, search.ExcludedEventCategories))));
            }

            if (search.IncludedEventTypes != null && search.IncludedEventTypes.Length > 0)
            {
                filters.Add(new FilterDescriptor<GroupEvent>().Terms(f => f.EventType, search.IncludedEventTypes));
            }

            if (search.ExcludedEventTypes != null && search.ExcludedEventTypes.Length > 0)
            {
                filters.Add(new FilterDescriptor<GroupEvent>().Bool(b => b.MustNot(bm => bm.Terms(f => f.EventType, search.ExcludedEventTypes))));
            }

            if (search.GiveawayID.HasValue)
            {
                filters.Add(new FilterDescriptor<GroupEvent>().Term(f => f.GiveawayDetails.GiveawayID, search.GiveawayID.Value));
            }

            if (!string.IsNullOrEmpty(search.UsernameQuery))
            {
                filters.Add(new FilterDescriptor<GroupEvent>().Bool(b => b.Should(
                    // Initiating user
                    bs => bs.Query(q => q.Match(qm => qm.OnField(t => t.InitiatingMemberUsername).Query(search.UsernameQuery))),
                    bs => bs.Query(q => q.Match(qm => qm.OnField(t => t.InitiatingMemberUsername.Suffix(CloudSearchConstants.AutocompleteSuffix)).Query(search.UsernameQuery))),

                    // Affected user
                    bs => bs.Query(q => q.Match(qm => qm.OnField(t => t.MemberDetails.Username).Query(search.UsernameQuery))),
                    bs => bs.Query(q => q.Match(qm => qm.OnField(t => t.MemberDetails.Username.Suffix(CloudSearchConstants.AutocompleteSuffix)).Query(search.UsernameQuery))),

                    // Giveaway Winner
                    bs => bs.Query(q => q.Match(qm => qm.OnField(t => t.GiveawayDetails.WinnerUsername).Query(search.UsernameQuery))),
                    bs => bs.Query(q => q.Match(qm => qm.OnField(t => t.GiveawayDetails.WinnerUsername.Suffix(CloudSearchConstants.AutocompleteSuffix)).Query(search.UsernameQuery)))
                    )));
            }

            // TODO: allow time specification
            var indexNames = new[] {IndexWildcard};
            //var indexNames = TimeSeriesIndexing.GetMonthsBetween(DateOfInception, DateTime.UtcNow).Select(m => TimeSeriesIndexing.GetIndexName(IndexTypeName, m.Year, m.Month));

            var size = search.PageSize ?? 10;
            size = size < 0 ? 0 : size;
            var skip = ((search.PageNumber ?? 1) - 1) * size;
            skip = skip < 0 ? 0 : skip;

            var searchDescriptor = new SearchDescriptor<GroupEvent>().Indices(indexNames)
                .Routing(search.GroupID.ToString())
                .Filter(f => f.Bool(b => b.Must(filters.ToArray())))
                .Size(size)
                .Skip(skip);

            var sortType = search.SortType ?? GroupEventSearchSortType.Timestamp;
            var sortAscending = search.SortAscending ?? sortType != GroupEventSearchSortType.Timestamp;
            switch (search.SortType ?? GroupEventSearchSortType.Timestamp)
            {
                case GroupEventSearchSortType.Timestamp:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.Timestamp).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case GroupEventSearchSortType.Category:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.EventCategory).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case GroupEventSearchSortType.EventType:
                    searchDescriptor = searchDescriptor.Sort(s => s.OnField(f => f.EventType).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                case GroupEventSearchSortType.InitiatingUsername:
                    searchDescriptor = searchDescriptor.Sort(s => s
                        .OnField(f => f.InitiatingMemberUsername.Suffix(CloudSearchConstants.SortSuffix))
                        .Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
            }

            var client = GetClient();
            var resp = client.Search<GroupEvent>(s => searchDescriptor);

            return resp.Documents.ToArray();
        }

        public static GroupEvent[] GetByGroupAndType(string groupID, GroupEventType type, int pageSize, int pageNumber)
        {
            var client = GetClient();

            var indexNames = new[] {IndexWildcard};

            var resp = client.Search<GroupEvent>(s => s.Filter(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.GroupID, groupID)))
                && q.Bool(b => b.Must(qd => qd.Term(ts => ts.EventType, (int) type))))
                .Size(pageSize)
                .Indices(indexNames)
                .Routing(groupID)
                .Skip((pageNumber - 1)*pageSize)
                .SortDescending(p => p.Timestamp));

            if (!resp.ConnectionStatus.Success)
            {
                Logger.Warn("Unable to retrieve group events: " + resp.ConnectionStatus.HttpStatusCode, new {groupID, type});
            }

            var requestParams = Encoding.Default.GetString(resp.RequestInformation.Request);
            Logger.Trace("Request Url: " + resp.RequestInformation.RequestUrl, new {Params = requestParams});
            return resp.Documents.ToArray();
        }

        public static void Index(GroupEvent groupEvent)
        {
            var client = GetClient();
            var timestamp = groupEvent.Timestamp.FromEpochMilliconds();
            var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, timestamp.Year, timestamp.Month);
            var resp = client.Index(groupEvent, idx => idx.Index(indexName).Routing(groupEvent.GroupID.ToString()));
            if (!resp.ConnectionStatus.Success)
            {
                throw new Exception("Failed to index group event: " + resp.ConnectionStatus.HttpStatusCode);
            }
        }

        public static void Index(GroupEvent[] groupEvents)
        {
            var client = GetClient();
            var operations = groupEvents.Select(e =>
            {
                var timeStamp = e.Timestamp.FromEpochMilliconds();
                return (IBulkOperation) new BulkCreateOperation<GroupEvent>(e)
                {
                    Routing = e.GroupID.ToString(),
                    Index = TimeSeriesIndexing.GetIndexName(IndexTypeName, timeStamp.Year, timeStamp.Month)
                };
            });
            var request = new BulkRequest
            {
                Operations = operations.ToArray()
            };
            var resp = client.Bulk(request);
            if (!resp.ConnectionStatus.Success)
            {
                throw new Exception("Failed to index group events");
            }
        }

        protected override PropertiesDescriptor<GroupEvent> CustomPropertyMapping(PropertiesDescriptor<GroupEvent> descriptor)
        {
            return descriptor
                .Object<GroupEventRootGroup>(n => n
                    .Name(g => g.PreviousRootGroupDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )
                .Object<GroupEventRootGroup>(n => n
                    .Name(g => g.RootGroupDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )

                .Object<GroupEventChannel>(n => n
                    .Name(g => g.PreviousChannelDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )
                .Object<GroupEventChannel>(n => n
                    .Name(g => g.ChannelDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )

                .Object<GroupEventRole>(n => n
                    .Name(g => g.PreviousRoleDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )
                .Object<GroupEventRole>(n => n
                    .Name(g => g.RoleDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )

                .Object<GroupEventLinkedCommunity>(o => o
                    .Name(g => g.PreviousLinkedCommunityDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )
                .Object<GroupEventLinkedCommunity>(o => o
                    .Name(g => g.LinkedCommunityDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )

                .Object<GroupEventMember>(n => n
                    .Name(g => g.MemberDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields()))

                .Object<GroupEventGiveaway>(o => o
                    .Name(g => g.GiveawayDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                )

                .Object<GroupEventPoll>(o => o
                    .Name(g => g.PollDetails)
                    .MapFromAttributes()
                    .Properties(p => p.CreateAutocompleteFields())
                );
        }

        #region Log Event Helpers

        public static void LogCreateGroupEvent(Group group, NewGroupMember creator)
        {
            GroupEventWorker.Create(GroupEvent.CreateRootGroup(group, creator));
        }

        public static void LogAddUsersEvent(Group group, NewGroupMember requestor, IEnumerable<NewGroupMember> added)
        {
            GroupEventWorker.Create(GroupEvent.ChangeMembership(group, requestor, added, GroupEventType.UsersAdded));
        }

        public static void LogAddUsersEvent(Group group, GroupMember requestor, IEnumerable<NewGroupMember> added)
        {
            GroupEventWorker.Create(GroupEvent.ChangeMembership(group, requestor, added, GroupEventType.UsersAdded));
        }

        public static void LogRemoveUsersEvent(Group group, GroupMember requestor, IEnumerable<GroupMember> removedMembers)
        {
            GroupEventWorker.Create(GroupEvent.ChangeMembership(group, requestor, removedMembers, GroupEventType.UsersRemoved));
        }

        public static void LogCreateSubgroup(Group parent, GroupMember requestor, Group subgroup, HashSet<int> accessRoles)
        {
            GroupEventWorker.Create(GroupEvent.CreateChannel(parent, subgroup, requestor, accessRoles.ToArray()));
        }

        public static void LogRemoveSubgroup(Group parent, GroupMember requestor, Group subgroup)
        {
            GroupEventWorker.Create(GroupEvent.DeleteChannel(parent, subgroup, requestor));
        }

        public static void LogRemoveUserRoleEvent(Group root, GroupMember requestor, GroupMember otherMember, GroupRole removed)
        {
            GroupEventWorker.Create(GroupEvent.ChangeUserRole(root, requestor, otherMember, removed, GroupEventType.UserRolesRemoved));
        }

        public static void LogAddUserRoleEvent(Group root, GroupMember requestor, GroupMember otherMember, GroupRole added)
        {
            GroupEventWorker.Create(GroupEvent.ChangeUserRole(root, requestor, otherMember, added, GroupEventType.UserRolesAdded));
        }

        public static void LogCreatePollEvent(GroupMember requestor, GroupPoll poll)
        {
            GroupEventWorker.Create(GroupEvent.FromPoll(requestor, poll, GroupEventType.PollStarted));
        }

        public static void LogEndPollEvent(GroupMember requestor, GroupPoll poll)
        {
            GroupEventWorker.Create(GroupEvent.FromPoll(requestor, poll, GroupEventType.PollEnded));
        }

        public static void LogCreateGiveawayEvent(GroupMember requestor, GroupGiveaway giveaway)
        {
            GroupEventWorker.Create(GroupEvent.FromGiveaway(requestor, giveaway, GroupEventType.GiveawayStarted));
        }

        public static void LogEndGiveawayEvent(GroupMember requestor, GroupGiveaway giveaway)
        {
            GroupEventWorker.Create(GroupEvent.FromGiveaway(requestor, giveaway, GroupEventType.GiveawayEnded));
        }

        public static void LogRollGiveawayEvent(GroupMember requestor, GroupGiveaway giveaway, GroupMember winner, GroupGiveawayRollStatus rollStatus, GroupRole winnerBestRole)
        {
            GroupEventWorker.Create(GroupEvent.FromGiveaway(requestor, giveaway, GroupEventType.GiveawayRoll, winner, rollStatus, winnerBestRole));
        }

        public static void LogMemberRenameEvent(Group group, int userID, string formerUsername, string newUsername)
        {
            GroupEventWorker.Create(GroupEvent.ChangeMemberNickname(group, userID, formerUsername, newUsername));
        }

        public static void LogExternalCommunityLinkedEvent(GroupMember requestor, ExternalCommunityMapping mapping)
        {
            GroupEventWorker.Create(GroupEvent.LinkExternalCommunity(requestor, mapping));
        }

        public static void LogExternalCommunityUnlinkedEvent(GroupMember requestor, ExternalCommunityMapping mapping)
        {
            GroupEventWorker.Create(GroupEvent.UnlinkExternalCommunity(requestor, mapping));
        }

        public static void LogExternalGuildLinkedEvent(GroupMember requestor, ExternalGuildMapping mapping)
        {
            GroupEventWorker.Create(GroupEvent.LinkExternalGuild(requestor, mapping));
        }

        public static void LogExternalGuildUnlinkedEvent(GroupMember requestor, ExternalGuildMapping mapping)
        {
            GroupEventWorker.Create(GroupEvent.UnlinkExternalGuild(requestor, mapping));
        }

        #endregion
    }
}
