﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Nest;

namespace Curse.Friends.Data.Search
{
    /// <summary>
    /// Represents an event that occurs within a group.
    /// </summary>
    [CloudSearchModel(AutoCreateIndex = false, UseDefaultIndex = false, IndexTypeName = "groupevents", UseAlias = true)]
    [ElasticType(Name = "groupEvent", IdProperty = "eventId")]
    public class LegacyGroupEvent
    {
        public LegacyGroupEvent()
        {
            EventID = Guid.NewGuid().ToString();
            Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
        }

        /// <summary>
        /// The ID (GUID) of the event.
        /// </summary>
        [ElasticProperty(Name = "eventId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string EventID { get; set; }

        /// <summary>
        /// The ID (GUID) of the root group.
        /// </summary>
        [ElasticProperty(Name = "groupId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public Guid GroupID { get; set; }

        /// <summary>
        /// The category (GroupEventCategory) of the event.
        /// </summary>
        [ElasticProperty(Name = "eventCategory", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public GroupEventCategory EventCategory { get; set; }

        /// <summary>
        /// The type (GroupEventType) of event.
        /// </summary>
        [ElasticProperty(Name = "type", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public LegacyNewGroupEventType EventType { get; set; }

        /// <summary>
        /// Time (Epoch Milliseconds) that the event occurred.
        /// </summary>
        [ElasticProperty(Name = "timestamp", Type = FieldType.Long, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public long Timestamp { get; set; }

        /// <summary>
        /// ID of the user that initiated the event.
        /// </summary>
        [ElasticProperty(Name = "initiatingUserId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int InitiatingUserID { get; set; }

        /// <summary>
        /// Name of the user that initiated the event.
        /// </summary>
        [AnalyzedStringProperty("initiatingUsername")]
        public string InitiatingUsername { get; set; }

        /// <summary>
        /// Users affected by the event.
        /// </summary>
        [ElasticProperty(Name = "affectedUsers", Type = FieldType.Nested, Store = true)]
        public GroupEventUser[] AffectedUsers { get; set; }

        /// <summary>
        /// Roles affected by the event.
        /// </summary>
        [ElasticProperty(Name = "affectedRoles", Type = FieldType.Nested, Store = true)]
        public GroupEventRole[] AffectedRoles { get; set; }

        /// <summary>
        /// Groups affected by the event.
        /// </summary>
        [ElasticProperty(Name = "affectedGroups", Type = FieldType.Nested, Store = true)]
        public GroupEventGroup[] AffectedGroups { get; set; }

        /// <summary>
        /// The affected giveaway.
        /// </summary>
        [ElasticProperty(Name = "affectedGiveaway", Type = FieldType.Object, Store = true)]
        public GroupEventGiveaway AffectedGiveaway { get; set; }

        /// <summary>
        /// The affected poll.
        /// </summary>
        [ElasticProperty(Name = "affectedPoll", Type = FieldType.Object, Store = true)]
        public GroupEventPoll AffectedPoll { get; set; }

        /// <summary>
        /// The affected linked community.
        /// </summary>
        [ElasticProperty(Name = "affectedLinkedCommunity", Type = FieldType.Object, Store = true)]
        public GroupEventLinkedCommunity AffectedLinkedCommunity { get; set; }

        /// <summary>
        /// Information about the roles that are affected by an event.
        /// </summary>
        public class GroupEventRole
        {
            /// <summary>
            /// The ID of a role affected by an event.
            /// </summary>
            [ElasticProperty(Name = "id", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int ID { get; set; }

            /// <summary>
            /// The name of a role affected by an event.
            /// </summary>
            [AnalyzedStringProperty("name")]
            public string Name { get; set; }

        }

        /// <summary>
        /// Information about the groups that are affected by an event.
        /// </summary>
        public class GroupEventGroup
        {
            /// <summary>
            /// The ID (GUID) of a group affected by an event.
            /// </summary>
            [ElasticProperty(Name = "id", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public Guid ID { get; set; }

            /// <summary>
            /// The type of a group (GroupType) affected by an event.
            /// </summary>
            [ElasticProperty(Name = "type", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.No)]
            public GroupType Type { get; set; }

            /// <summary>
            /// The title of a group affected by a rename event.
            /// </summary>
            [AnalyzedStringProperty("title")]
            public string Title { get; set; }

            /// <summary>
            /// The ID (GUID) of an affected group's parent.
            /// </summary>
            [ElasticProperty(Name = "parentId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public Guid ParentID { get; set; }

            /// <summary>
            /// The title of an affected group's parent.
            /// </summary>
            [AnalyzedStringProperty("parentTitle")]
            public string ParentTitle { get; set; }

        }

        /// <summary>
        /// Information about the users that are affected by an event.
        /// </summary>
        public class GroupEventUser
        {
            /// <summary>
            /// The ID of a user affected by an event.
            /// </summary>
            [ElasticProperty(Name = "id", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int ID { get; set; }

            /// <summary>
            /// The username of a user affected by an event.
            /// </summary>
            [AnalyzedStringProperty("username")]
            public string Username { get; set; }

            #region Creation Helper Methods

            public static GroupEventUser FromGroupMember(GroupMember member)
            {
                return new GroupEventUser
                {
                    ID = member.UserID,
                    Username = member.GetTitleName()
                };
            }

            public static GroupEventUser FromNewGroupMember(NewGroupMember member)
            {
                return new GroupEventUser
                {
                    ID = member.UserID,
                    Username = member.Username
                };
            }

            #endregion

            public GroupEventUserContract ToContract()
            {
                return new GroupEventUserContract
                {
                    UserID = ID,
                    Username = Username
                };
            }
        }

        public class GroupEventGiveaway
        {
            [ElasticProperty(Name = "id", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int GiveawayID { get; set; }

            [AnalyzedStringProperty("title")]
            public string Title { get; set; }

            [ElasticProperty(Name = "totalEntries", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int TotalEntries { get; set; }

            [ElasticProperty(Name = "validWinner", Type = FieldType.Boolean, Store = true, Index = FieldIndexOption.No)]
            public bool ValidWinner { get; set; }

            [ElasticProperty(Name = "claimedPrize", Type = FieldType.Boolean, Store = true, Index = FieldIndexOption.No)]
            public bool ClaimedPrize { get; set; }

            [ElasticProperty(Name = "rollNumber", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.No)]
            public int RollNumber { get; set; }

        }

        public class GroupEventPoll
        {
            [ElasticProperty(Name = "id", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int ID { get; set; }

            [AnalyzedStringProperty("title")]
            public string Title { get; set; }

            [ElasticProperty(Name = "totalVotes", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int TotalVotes { get; set; }

            [ElasticProperty(Name = "requiredRoles", Store = true, Index = FieldIndexOption.No)]
            public int[] RequiredRoles { get; set; }

            [ElasticProperty(Name = "totalOptions", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.No)]
            public int TotalOptions { get; set; }

            [AnalyzedStringProperty("winningOptions")]
            public string[] WinningOptions { get; set; }

            [ElasticProperty(Name = "durationMinutes", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
            public int DurationMinutes { get; set; }

        }

        public class GroupEventLinkedCommunity
        {
            [ElasticProperty(Name = "id", Type = FieldType.String, Store = true, Index = FieldIndexOption.Analyzed)]
            public string ID { get; set; }

            [AnalyzedStringProperty("externalName")]
            public string ExternalName { get; set; }

            [AnalyzedStringProperty("externalDisplayName")]
            public string ExternalDisplayName { get; set; }

        }
    }
}
