﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Friends.Enums;
using Curse.Logging;
using Elasticsearch.Net;
using Nest;
using Curse.Extensions;

namespace Curse.Friends.Data.Search
{
    public class GroupMemberManager : CloudSearchManager<GroupMemberSearchModel>
    {
        private static readonly LogCategory Logger = new LogCategory("GroupMemberManager");

        public static GroupMemberSearchModel[] SearchGroupMembers(Guid groupID, GroupMemberSearch search)
        {
            var client = GetClient();

            var size = search.PageSize;
            var skip = (search.PageNumber - 1) * search.PageSize;

            var queries = new List<QueryContainer>();
            var mustFilters = new List<FilterContainer>();
            var shouldFilters = new List<FilterContainer>();

            // Musts
            mustFilters.Add(new FilterDescriptor<GroupMemberSearchModel>().Term(t => t.GroupID, groupID));
            mustFilters.Add(new FilterDescriptor<GroupMemberSearchModel>().Term(t => t.IsDeleted, false));
            if (search.RoleID.HasValue)
            {
                mustFilters.Add(new FilterDescriptor<GroupMemberSearchModel>().Term(t => t.Roles, search.RoleID.Value));
            }

            // Query Searches
            if (!string.IsNullOrWhiteSpace(search.Username))
            {
                queries.Add(new QueryDescriptor<GroupMemberSearchModel>().Bool(b => b.Should(bs =>
                    bs.Term(g => g.Username.Suffix(AutocompleteSuffix), search.Username.ToLowerInvariant()) || bs.Term(g => g.Nickname.Suffix(AutocompleteSuffix), search.Username.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(g => g.Username).Query(search.Username)) || bs.Match(m => m.OnField(g => g.Nickname).Query(search.Username))
                    )));
            }

            var searchDescriptor = new SearchDescriptor<GroupMemberSearchModel>()
                .Query(q => q.Bool(b => b.Should(queries.ToArray())))
                .Filter(f =>
                    f.Bool(b => b.Must(mustFilters.ToArray())) &&
                    f.Bool(b => b.Should(shouldFilters.ToArray())))
                .Index(IndexName)
                .Routing(groupID.ToString())
                .Skip(skip)
                .Size(size);

            var sortAscending = search.SortAscending ?? true;
            switch (search.SortType)
            {
                case GroupMemberSearchSortType.Role:
                    searchDescriptor = sortAscending ? searchDescriptor.SortAscending(g => g.BestRoleRank) : searchDescriptor.SortDescending(g => g.BestRoleRank);
                    break;
                case GroupMemberSearchSortType.DateJoined:
                    searchDescriptor = sortAscending ? searchDescriptor.SortAscending(g => g.DateJoined) : searchDescriptor.SortDescending(g => g.DateJoined);
                    break;
                case GroupMemberSearchSortType.DateLastActive:
                    searchDescriptor = sortAscending ? searchDescriptor.SortAscending(g => g.DateLastActive) : searchDescriptor.SortDescending(g => g.DateLastActive);
                    break;
                case GroupMemberSearchSortType.Username:
                case GroupMemberSearchSortType.Default:
                    searchDescriptor = searchDescriptor.Sort(sd => sd.OnField(g => g.Username.Suffix(SortSuffix)).Order(sortAscending ? SortOrder.Ascending : SortOrder.Descending));
                    break;
                default:
                    if (string.IsNullOrWhiteSpace(search.Username))
                    {
                        // If it isn't a text search, sort by role ascending - text searches are sorted by score
                        searchDescriptor = searchDescriptor.SortAscending(gm => gm.BestRole);
                    }
                    break;
            }

            var resp = client.Search<GroupMemberSearchModel>(searchDescriptor);

            if (!resp.ConnectionStatus.Success)
            {
                return new GroupMemberSearchModel[0];
            }

            return resp.Hits.Select(h => h.Source).ToArray();
        }

        public static Dictionary<int, long> CountMembersByRoles(Guid groupID)
        {
            var client = GetClient();

            var results = client.Search<GroupMemberSearchModel>(s => s
                .Index(IndexName)
                .Routing(groupID.ToString())
                .SearchType(SearchType.Count)
                .Query(q => q.Term(t => t.GroupID, groupID) && q.Term(t => t.IsDeleted, false))
                .Aggregations(a => a.Terms("roles", d => d.Field(f => f.Roles))));

            return results.Aggs.Terms("roles").Items.ToDictionary(d => int.Parse(d.Key), d => d.DocCount);
        }

        public static GroupMemberSearchModel[] UserMembersByRoles(Guid groupID, HashSet<int> roles)
        {
            var client = GetClient();

            var results = client.Search<GroupMemberSearchModel>(s => s
                .Index(IndexName)
                .Routing(groupID.ToString())
                .SearchType(SearchType.Count)
                .Query(
                    q =>
                        q.Term(t => t.GroupID, groupID)
                        && q.Term(t => t.IsDeleted, false) 
                        && q.Terms(t => t.Roles, roles.ToArray())));

            if (!results.ConnectionStatus.Success)
            {
                Logger.Warn("Failed to get members by roles", results.ConnectionStatus);
            }

            return results.Documents.ToArray();

        }

        public static GroupMemberSearchModel[] MembersByUsernameOrNickname(Guid groupID, string username)
        {
            var client = GetClient();

            var results = client.Search<GroupMemberSearchModel>(s => s
                .Index(IndexName)
                .Routing(groupID.ToString())
                .Query(q => q.Term(t => t.GroupID, groupID)
                            && (q.Match(m => m.OnField(g => g.Username).Query(username))
                                || q.Match(m => m.OnField(g => g.Nickname).Query(username)))
                ));

            if (!results.ConnectionStatus.Success)
            {
                Logger.Warn("Failed to get members by username or nickname", results.ConnectionStatus);
                throw new Exception("Failed to get group members by username or nickname.");
            }

            return results.Documents.ToArray();
        }

        public static GroupMemberSearchModel[] GetInactiveByGroup(Guid groupID, int? pageSize = null, int? pageNumber = null)
        {
            var client = GetClient();
            var results = new List<GroupMemberSearchModel>();

            var size = pageSize ?? 50;
            var page = pageNumber ?? 1;
            if (page == 0)
            {
                page = 1;
            }

            var activityTimestamp = DateTime.UtcNow.AddMinutes(-10).ToEpochMilliseconds();

            var resp = client.Search<GroupMemberSearchModel>(s => s
                .Filter(f =>
                    f.Term(t => t.GroupID, groupID.ToString())
                    && f.Term(t => t.IsDeleted, false)
                   && f.Range(r => r.OnField(d => d.DateLastActive).Lower(activityTimestamp)))
                .Index(IndexName)
                .Routing(groupID.ToString())
                .Sort(p => p.OnField(f => f.Username).Ascending())
                .Size(size)
                .Skip((page - 1) * size));

            if (!resp.ConnectionStatus.Success)
            {
                return new GroupMemberSearchModel[0];
            }


            return resp.Documents.ToArray();
        }

        public static GroupMemberSearchModel[] GetByGroup(Guid groupID, int? pageSize = null, int? pageNumber = null)
        {
            var client = GetClient();

            var size = pageSize ?? 50;
            var page = pageNumber ?? 1;
            if (page == 0)
            {
                page = 1;
            }

            var activityTimestamp = DateTime.UtcNow.AddMinutes(-10).ToEpochMilliseconds();

            var resp = client.Search<GroupMemberSearchModel>(s => s
                .Filter(f =>
                    f.Term(t => t.GroupID, groupID.ToString())
                    && f.Term(t => t.IsDeleted, false)
                   && f.Range(r => r.OnField(d => d.DateLastActive).GreaterOrEquals(activityTimestamp)))
                .Index(IndexName)
                .Routing(groupID.ToString())
                .Sort(p => p.OnField(f => f.BestRoleRank).Ascending())
                .Sort(p => p.OnField(f => f.Username).Ascending())
                .Size(size)
                .Skip((page - 1) * size));


            if (!resp.ConnectionStatus.Success)
            {
                return new GroupMemberSearchModel[0];
            }


            return resp.Documents.ToArray();
        }

        private static GroupMemberSearchModel GetMember(Guid groupID, int userID)
        {
            var client = GetClient();

            var id = GroupMemberSearchModel.CreateModelID(groupID, userID);

            var resp = client.Get<GroupMemberSearchModel>(g => g
                .Id(id)
                .Index(IndexName)
                .Routing(groupID.ToString()));

            if (!resp.Found || resp.Source == null)
            {
                return null;
            }

            if (resp.Source.GroupID != groupID || resp.Source.UserID != userID)
            {
                return null;
            }

            return resp.Source;
        }

        public static void AddOrUpdateMember(GroupMember member)
        {
            try
            {
                var existingMember = GetMember(member.GroupID, member.UserID);
                if (existingMember == null)
                {
                    Index(GroupMemberSearchModel.FromGroupMember(member));
                }
                else
                {
                    UpdateMember(existingMember, GroupMemberSearchModel.FromGroupMember(member));
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while adding member", new { member });
            }
        }

        public static void BatchUpsertMember(IEnumerable<GroupMemberSearchModel> members)
        {
            var client = GetClient();

            foreach (var batch in members.InSetsOf(1000))
            {
                var request = new BulkRequest
                {
                    Index = IndexName,
                    Operations = batch.Select(m => (IBulkOperation)new BulkUpdateDescriptor<GroupMemberSearchModel, GroupMemberSearchModel>()
                        .Doc(m).DocAsUpsert().Id(m.ModelID).Routing(m.GroupID.ToString())).ToArray()
                };

                var resp = client.Bulk(request);

                if (!resp.ConnectionStatus.Success)
                {
                    throw new Exception("Failed to bulk upsert members: " + resp.ConnectionStatus.HttpStatusCode);
                }
            }

        }

        public static IEnumerable<IEnumerable<GroupMemberSearchModel>> ScrollMembers(Guid groupID)
        {
            var client = GetClient();

            var response = client.Search<GroupMemberSearchModel>(s => s.Filter(f => f.Term(t => t.GroupID, groupID.ToString())).Routing(groupID.ToString()).Scroll("30s"));
            while (response.Documents.Any())
            {
                yield return response.Documents;
                response = client.Scroll<GroupMemberSearchModel>(s => s.Scroll("30s").ScrollId(response.ScrollId));
            }
        }

        public static void Index(GroupMemberSearchModel groupMemberSearch)
        {
            var client = GetClient();
            var resp = client.Index(groupMemberSearch, idx => idx.Index(IndexName).Routing(groupMemberSearch.GroupID.ToString()));
            if (!resp.ConnectionStatus.Success)
            {
                throw new Exception("Failed to index group member: " + resp.ConnectionStatus.HttpStatusCode);
            }
        }

        public static void UpdateActivityDate(Guid groupID, int[] userIDs, DateTime timestamp)
        {
            var client = GetClient();
            var parameters = new FluentDictionary<string, object> { { "dateLastActive", timestamp.ToEpochMilliseconds() } };

            foreach (var batch in userIDs.InSetsOf(1000))
            {
                try
                {
                    var request = new BulkRequest
                    {
                        Index = IndexName,
                        Operations = batch.Select(userID => (IBulkOperation)new BulkUpdateDescriptor<GroupMemberSearchModel, GroupMemberSearchModel>()
                            .Script("ctx._source.dateLastActive = dateLastActive;")
                            .Params(p => parameters)
                            .Id(GroupMemberSearchModel.CreateModelID(groupID, userID))
                            .Routing(groupID.ToString()))
                        .ToArray()
                    };

                    var resp = client.Bulk(request);

                    if (!resp.ConnectionStatus.Success)
                    {
                        Logger.Warn("Failed to update activity date for set!");
                    }
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Failed to update activity date for set!");
                }
                
            }
        }

        public static void UpdateUsername(Guid groupID, int userID, string username, string nickname, string displayName)
        {
            var client = GetClient();

            var resp = client.Update<GroupMemberSearchModel>(u => u
                            .Id(GroupMemberSearchModel.CreateModelID(groupID, userID))
                            .Index(IndexName)
                            .Routing(groupID.ToString())
                            .Script("ctx._source.username = username;ctx._source.nickname = nickname;ctx._source.displayName = displayName;")
                            .Params(p => p.Add("username", username).Add("nickname", nickname.ToEmptyWhenNull()).Add("displayName", displayName.ToEmptyWhenNull()))
                            .RetryOnConflict(3)
                            .Refresh());

            if (!resp.ConnectionStatus.Success)
            {
                throw new Exception("Failed to update member name: " + resp.ConnectionStatus.HttpStatusCode);
            }

        }

        private static bool ShouldUpdate(GroupMemberSearchModel original, GroupMemberSearchModel updated, out List<string> scriptLines, out FluentDictionary<string, object> parameters)
        {
            parameters = new FluentDictionary<string, object>();
            scriptLines = new List<string>();

            if (original.Username != updated.Username)
            {
                parameters.Add("username", updated.Username.ToEmptyWhenNull());
                scriptLines.Add("ctx._source.username = username;");
            }

            if (original.Nickname != updated.Nickname)
            {
                parameters.Add("nickname", updated.Nickname.ToEmptyWhenNull());
                scriptLines.Add("ctx._source.nickname = nickname;");
            }

            if (updated.DisplayName != null)
            {
                if (original.DisplayName != updated.DisplayName)
                {
                    parameters.Add("displayName", updated.DisplayName.ToEmptyWhenNull());
                    scriptLines.Add("ctx._source.displayName = displayName;");
                }
            }
           

            if (!original.Roles.SequenceEqual(updated.Roles))
            {
                parameters.Add("roles", updated.Roles);
                scriptLines.Add("ctx._source.roles = roles;");
            }

            if (original.DateJoined != updated.DateJoined)
            {
                parameters.Add("dateJoined", updated.DateJoined);
                scriptLines.Add("ctx._source.dateJoined = dateJoined;");
            }

            if (original.BestRole != updated.BestRole)
            {
                parameters.Add("bestRole", updated.BestRole);
                scriptLines.Add("ctx._source.bestRole = bestRole;");
            }

            if (original.BestRoleRank != updated.BestRoleRank)
            {
                parameters.Add("BestRoleRank", updated.BestRoleRank);
                scriptLines.Add("ctx._source.BestRoleRank = BestRoleRank;");
            }

            if (original.IsDeleted != updated.IsDeleted)
            {
                parameters.Add("isDeleted", updated.IsDeleted);
                scriptLines.Add("ctx._source.isDeleted = isDeleted;");
            }

            if (original.DateLastActive != updated.DateLastActive)
            {
                parameters.Add("dateLastActive", updated.DateLastActive);
                scriptLines.Add("ctx._source.dateLastActive = dateLastActive;");
            }

            return scriptLines.Count > 0;
        }

        private static void UpdateMember(GroupMemberSearchModel original, GroupMemberSearchModel updated)
        {
            FluentDictionary<string, object> parameters;
            List<string> scriptLines;
            if (ShouldUpdate(original, updated, out scriptLines, out parameters))
            {
                var client = GetClient();
                var id = GroupMemberSearchModel.CreateModelID(updated.GroupID, updated.UserID);
                var resp = client.Update<GroupMemberSearchModel>(u => u
                    .Id(id)
                    .Index(IndexName)
                    .Routing(updated.GroupID.ToString())
                    .Script(string.Join("", scriptLines))
                    .Params(p => parameters)
                    .RetryOnConflict(3)
                    .Refresh());

                if (!resp.ConnectionStatus.Success || !resp.IsValid)
                {
                    Logger.Warn("Failed to update group member", new { StatusCode = resp.ConnectionStatus.HttpStatusCode, original, updated });
                }
            }
        }

        public static void DeleteMember(GroupMember member)
        {
            var client = GetClient();

            try
            {
                var id = GroupMemberSearchModel.CreateModelID(member.GroupID, member.UserID);
                var resp = client.Update<GroupMemberSearchModel>(u => u
                    .Id(id)
                    .Index(IndexName)
                    .Routing(member.GroupID.ToString())
                    .Script("ctx._source.isDeleted = isDeleted;")
                    .Params(p => p
                        .Add("isDeleted", true))
                    .RetryOnConflict(3)
                    .Refresh());                

                if (!resp.IsValid)
                {
                    if (resp.ConnectionStatus != null && 
                        resp.ConnectionStatus.HttpStatusCode.HasValue &&
                        resp.ConnectionStatus.HttpStatusCode.Value == 404)
                    {
                        Logger.Info("Member not found in index.");
                    }
                    else
                    {
                        Logger.Error("Failed to delete member from index", resp.ConnectionStatus);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while deleting a member", new { member });
            }
        }
    }
}
