﻿using System;
using System.Linq;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Nest;

namespace Curse.Friends.Data.Search
{
    [ElasticType(Name = "groupMember", IdProperty = "modelId")]
    [CloudSearchModel(AutoCreateIndex = false, UseDefaultIndex = false, IndexTypeName = "groupmembers")]
    public class GroupMemberSearchModel
    {
        [ElasticProperty(Name="modelId", Type = FieldType.String, Store=true, Index = FieldIndexOption.NotAnalyzed)]
        public string ModelID { get; set; }

        [ElasticProperty(Name = "groupId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public Guid GroupID { get; set; }

        [ElasticProperty(Name = "userId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int UserID{get;set;}

        [AnalyzedStringProperty("username")]
        public string Username { get; set; }

        [AnalyzedStringProperty("displayName")]
        public string DisplayName { get; set; }

        [AnalyzedStringProperty("nickname")]
        public string Nickname { get; set; }

        [ElasticProperty(Name = "regionId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int RegionID { get; set; }

        [ElasticProperty(Name = "isDeleted", Type = FieldType.Boolean, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public bool IsDeleted { get; set; }

        [ElasticProperty(Name = "bestRole", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int BestRole { get; set; }

        [ElasticProperty(Name="bestRoleRank", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int BestRoleRank { get; set; }

        [ElasticProperty(Name = "roles", Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int[] Roles { get; set; }

        [ElasticProperty(Name = "dateJoined", Type = FieldType.Long, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public long DateJoined { get; set; }

        [ElasticProperty(Name = "dateLastActive", Type = FieldType.Long, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public long DateLastActive { get; set; }

        public static string CreateModelID(Guid groupID, int userID)
        {
            return groupID + ":" + userID;
        }

        public static GroupMemberSearchModel FromGroupMember(GroupMember member)
        {
            return new GroupMemberSearchModel
            {
                ModelID = CreateModelID(member.GroupID, member.UserID),
                GroupID = member.GroupID,
                UserID = member.UserID,
                Username = member.Username,
                Nickname = member.Nickname,
                RegionID = member.RegionID,
                Roles = member.Roles.ToArray(),
                BestRole = member.BestRole,
                BestRoleRank = member.BestRoleRank,
                IsDeleted = member.IsDeleted,
                DateJoined = member.DateJoined.ToEpochMilliseconds(),
                DateLastActive = member.DateLastActive.ToEpochMilliseconds(),
                DisplayName = member.DisplayName
            };
        }        

        public GroupMemberContract ToSearchNotification(UserStatistics userStats)
        {
            return new GroupMemberContract
            {
                UserID = UserID,
                Username =  userStats != null ? userStats.Username ?? Username : Username,
                DisplayName = userStats != null ? userStats.DisplayName ?? DisplayName : DisplayName,
                Nickname = Nickname,
                Roles = Roles,
                BestRole = BestRole,
                DateJoined = DateJoined,
                ConnectionStatus = userStats != null ? userStats.ConnectionStatus : UserConnectionStatus.Offline,
                DateLastActive = DateLastActive,
                DateLastSeen = userStats != null ? userStats.DateLastSeen : 0,
                IsActive = DateLastActive > 0 && DateTime.UtcNow.ToEpochMilliseconds() - DateLastActive < GroupMember.ActivityMilliseconds,
                CurrentGameID = userStats != null ? userStats.CurrentGameID : 0,
                AvatarTimestamp = userStats != null ? userStats.AvatarTimestamp : 0
            };
        }
    }
}
