﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudSearch;
using Curse.Friends.Enums;
using Curse.Logging;
using Nest;

namespace Curse.Friends.Data.Search
{
    public class GroupSearchManager : CloudSearchManager<GroupSearchModel>
    {
        private static readonly string _indexName = "groups-index";

        private static readonly int _defaultPageSize = 20;
        public static readonly int MaxPageSize = 100;

        public static void Index(GroupSearchModel model)
        {
            var client = GetClient();

            var res = client.Index(model, i => i.Index(_indexName));

            if (!res.ConnectionStatus.Success)
            {
                Logger.Error("Failed to index a group for search", model);
                throw new InvalidOperationException("Failed to index group for search");
            }
        }

        public static void UpdateMemberCount(Guid groupID, int memberCount, int newQuality)
        {
            var client = GetClient();

            var updateDescriptor = new UpdateDescriptor<GroupSearchModel, GroupSearchModel>()
                .Id(groupID.ToString())
                .Index(_indexName)
                .Script("ctx._source.memberCount=memberCount;ctx._source.quality=quality;")
                .Params(p => p
                    .Add("memberCount", memberCount)
                    .Add("quality", newQuality)
                );
          
            var res = client.Update<GroupSearchModel>(u => updateDescriptor);

            if (!res.ConnectionStatus.Success)
            {
                Logger.Trace("Failed to update a group's info", new { groupID, memberCount });
            }
        }

        public static void UpdateStreamingStatus(Guid groupID, bool isStreaming, long streamingTimestamp)
        {
            var client = GetClient();

            var updateDescriptor = new UpdateDescriptor<GroupSearchModel, GroupSearchModel>()
                .Id(groupID.ToString())
                .Index(_indexName)
                .Script("ctx._source.isStreaming=isStreaming;ctx._source.streamingTimestamp=streamingTimestamp;")
                .Params(p => p
                    .Add("isStreaming", isStreaming)
                    .Add("streamingTimestamp", streamingTimestamp)
                 );
         
            var res = client.Update<GroupSearchModel>(u => updateDescriptor);

            if (!res.ConnectionStatus.Success)
            {
                Logger.Trace("Failed to update a group's streaming status", new { groupID });
            }
        }

        public static void UpdateFeaturedStatus(Guid groupID, bool isFeatured, long featuredTimestamp)
        {
            var client = GetClient();

            var updateDescriptor = new UpdateDescriptor<GroupSearchModel, GroupSearchModel>()
                .Id(groupID.ToString())
                .Index(_indexName)
                .Script("ctx._source.isFeatured=isFeatured;ctx._source.featuredTimestamp=featuredTimestamp;")
                .Params(p => p
                    .Add("isFeatured", isFeatured)
                    .Add("featuredTimestamp", featuredTimestamp)
                );

            var res = client.Update<GroupSearchModel>(u => updateDescriptor);

            if (!res.ConnectionStatus.Success)
            {
                Logger.Trace("Failed to update a group's featured status", new {groupID});
            }
        }

        public static void UpdateSearchability(Guid groupID, bool isSearchable)
        {
            var client = GetClient();

            var updateDescriptor = new UpdateDescriptor<GroupSearchModel, GroupSearchModel>()
                .Id(groupID.ToString())
                .Index(_indexName)
                .Script("ctx._source.isDeleted=isDeleted;")
                .Params(p => p.Add("isDeleted", !isSearchable));
            
            var res = client.Update<GroupSearchModel>(u => updateDescriptor);

            if (!res.ConnectionStatus.Success)
            {
                Logger.Trace("Failed to update a group's searchability", new { groupID, isSearchable });
            }
        }

        public static void UpdateQuarantineStatus(Guid groupId, bool flaggedAsInappropriate)
        {
            var client = GetClient();

            var res = client.Update<GroupSearchModel>(u => u
                .Id(groupId.ToString())
                .Index(IndexName)
                .Script("ctx._source.flaggedAsInappropriate=flaggedAsInappropriate;")
                .Params(p => p.Add("flaggedAsInappropriate", flaggedAsInappropriate))
                );

            if (!res.ConnectionStatus.Success)
            {
                Logger.Trace("Failed to update a group's flag for inappropriate content");
            }
        }


        public static GroupSearchModel[] Search(GroupSearch search)
        {
            var queries = new List<QueryContainer> { new QueryDescriptor<GroupSearchModel>().Term(f => f.IsDeleted, false) };

            if (search.IsPublic.HasValue)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Term(f => f.IsPublic, search.IsPublic.Value));
            }

            if (search.GroupID.HasValue)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Term(f => f.GroupID, search.GroupID.Value));
            }

            if (!string.IsNullOrWhiteSpace(search.GroupTitle))
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Match(m => m.OnField(f => f.GroupTitle).Query(search.GroupTitle)));
            }

            if (!string.IsNullOrWhiteSpace(search.OwnerUsername))
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Match(m => m.OnField(f => f.OwnerUsername).Query(search.OwnerUsername)));
            }

            if (search.Tags != null && search.Tags.Length > 0)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Terms(f => f.Tags, search.Tags));
            }

            if (!search.IncludeInappropriate)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Bool(b=>b.MustNot(bm=>bm.Term(t=>t.FlaggedAsInappropriate, true))));
            }

            if (!string.IsNullOrWhiteSpace(search.Query))
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Bool(b => b.Should(
                    bs => bs.Term(t => t.GroupTitle.Suffix(AutocompleteSuffix), search.Query.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.GroupTitle).Query(search.Query)),
                    bs => bs.Term(t => t.OwnerUsername.Suffix(AutocompleteSuffix), search.Query.ToLowerInvariant()),
                    bs => bs.Match(m => m.OnField(f => f.OwnerUsername).Query(search.Query)),
                    bs => bs.Match(m => m.OnField(f => f.Description).Query(search.Query))
                    )));
            }

            if (search.IsFeatured.HasValue)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Term(f=>f.IsFeatured,search.IsFeatured.Value));
            }

            if (search.IsStreaming.HasValue)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Term(f => f.IsStreaming, search.IsStreaming.Value));
            }

            if (search.MinMemberCount.HasValue || search.MaxMemberCount.HasValue)
            {
                if (search.MinMemberCount.HasValue && search.MaxMemberCount.HasValue && search.MinMemberCount.Value > search.MaxMemberCount.Value)
                {
                    throw new InvalidOperationException("Min member count cannot exceed max member count!");
                }

                queries.Add(new QueryDescriptor<GroupSearchModel>().Range(r =>
                {
                    var descriptor = r.OnField(f => f.MemberCount);
                    if (search.MinMemberCount.HasValue)
                    {
                        descriptor = descriptor.GreaterOrEquals(search.MinMemberCount.Value);
                    }
                    if (search.MaxMemberCount.HasValue)
                    {
                        descriptor.LowerOrEquals(search.MaxMemberCount.Value);
                    }
                }));
            }

            if (search.Games != null && search.Games.Length > 0)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Bool(b => b.Should(bs => bs.Terms(f => f.Games, search.Games), bs => bs.Term(f => f.MatchAllGames, true))));
            }

            if (search.SubType.HasValue)
            {
                queries.Add(new QueryDescriptor<GroupSearchModel>().Term(g => g.Subtype, search.SubType.Value));
            }

            var client = GetClient();

            Func<SortFieldDescriptor<GroupSearchModel>, IFieldSort> sortFunc;
            switch (search.SortType)
            {
                case GroupSearchSortType.MemberCount:
                    sortFunc = descriptor => descriptor.OnField(f => f.MemberCount).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending);
                    break;
                case GroupSearchSortType.DateFeatured:
                    sortFunc = descriptor => descriptor.OnField(f => f.FeaturedTimestamp).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending);
                    break;
                case GroupSearchSortType.DateStreaming:
                    sortFunc = descriptor => descriptor.OnField(f => f.StreamingTimestamp).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending);
                    break;
                case GroupSearchSortType.Title:
                    sortFunc = descriptor => descriptor.OnField(f => f.GroupTitle.Suffix(SortSuffix)).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending);
                    break;
                case GroupSearchSortType.Default:
                default:
                    sortFunc = descriptor => descriptor.OnField(f => f.Quality).Order(search.SortAscending ? SortOrder.Ascending : SortOrder.Descending);
                    break;
            }

            var size = search.PageSize ?? _defaultPageSize;
            size = size < 1 ? 1 : size;
            size = size > MaxPageSize ? MaxPageSize : size;
            var skip = ((search.PageNumber ?? 1) - 1) * size;
            var result = client.Search<GroupSearchModel>(s => s
                .Query(q => q.Bool(b => b
                    .Must(queries.ToArray())))
                .Index(_indexName)
                .Size(size)
                .Skip(skip)
                .Sort(sortFunc));

            if (!result.ConnectionStatus.Success)
            {
                Logger.Warn("Failed to run group search", search);
            }

            return result.Documents.ToArray();
        }
    }
}
