﻿using System;
using System.Linq;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Nest;

namespace Curse.Friends.Data.Search
{
    [CloudSearchModel(AutoCreateIndex = false, UseDefaultIndex = false, IndexTypeName = "groups")]
    [ElasticType(Name = "group", IdProperty = "groupId")]
    public class GroupSearchModel
    {
        [ElasticProperty(Name = "groupId", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.String)]
        public Guid GroupID { get; set; }

        [AnalyzedStringProperty("groupTitle")]
        public string GroupTitle { get; set; }

        [ElasticProperty(Name = "ownerUserId", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Integer)]
        public int OwnerUserID { get; set; }

        [AnalyzedStringProperty("ownerUsername")]
        public string OwnerUsername { get; set; }

        [ElasticProperty(Name = "tags", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Integer)]
        public GroupSearchTag[] Tags { get; set; }

        [ElasticProperty(Name = "isPublic", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Boolean)]
        public bool IsPublic { get; set; }

        [ElasticProperty(Name = "description", Store = true, Index = FieldIndexOption.Analyzed, Type = FieldType.String)]
        public string Description { get; set; }

        [ElasticProperty(Name = "memberCount", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Integer)]
        public int MemberCount { get; set; }

        [ElasticProperty(Name = "games", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Integer)]
        public int[] Games { get; set; }

        [ElasticProperty(Name = "matchAllGames", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Boolean)]
        public bool MatchAllGames { get; set; }

        [ElasticProperty(Name = "isDeleted", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Boolean)]
        public bool IsDeleted { get; set; }

        [ElasticProperty(Name = "subtype", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Integer)]
        public GroupSubType Subtype { get; set; }

        [ElasticProperty(Name = "isStreaming", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Boolean)]
        public bool IsStreaming { get; set; }

        [ElasticProperty(Name = "streamingTimestamp", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Long)]
        public long StreamingTimestamp { get; set; }

        [ElasticProperty(Name = "isFeatured", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Boolean)]
        public bool IsFeatured { get; set; }

        [ElasticProperty(Name = "featuredTimestamp", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Long)]
        public long FeaturedTimestamp { get; set; }

        [ElasticProperty(Name = "quality", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Integer)]
        public int Quality { get; set; }

        [ElasticProperty(Name = "flaggedAsInappropriate", Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Boolean)]
        public bool FlaggedAsInappropriate { get; set; }

        [ElasticProperty(Name = "creationTimestamp", Store = true, Index = FieldIndexOption.NotAnalyzed, Type = FieldType.Long)]
        public long CreationTimestamp { get; set; }

        public GroupSearchModelContract ToNotification()
        {
            return new GroupSearchModelContract
            {
                Description = Description,
                GroupID = GroupID,
                GroupTitle = GroupTitle,
                IsPublic = IsPublic,
                MemberCount = MemberCount,
                OwnerUserID = OwnerUserID,
                OwnerUsername = OwnerUsername,
                Tags = Tags,
                Games = Games,
                MatchAllGames = MatchAllGames,
                IsStreaming = IsStreaming,
                StreamingTimestamp = StreamingTimestamp,
                Subtype = Subtype,
                IsFeatured = IsFeatured,
                FeaturedTimestamp = FeaturedTimestamp,
                FlaggedAsInappropriate = FlaggedAsInappropriate,
                CreationTimestamp = CreationTimestamp,
                SortRank = Quality
            };
        }

        public static int CalculateQuality(Group group, GroupSearchSettings searchSettings, Avatar avatar, ExternalCommunity community = null)
        {
            // Groups without members should never be rated higher than groups with members, regardless of other factors
            if (group.MemberCount == 0)
            {
                return 0;
            }

            // Baseline for quality is member count
            var memberCount = group.MemberCount;
            if (community != null)
            {
                memberCount += community.Followers;
            }
            var quality =  (int) Math.Log(memberCount + 1)*2;

            if (!string.IsNullOrWhiteSpace(searchSettings.Description))
            {
                quality += 2;
            }

            if (searchSettings.MatchAllGames || (searchSettings.Games != null && searchSettings.Games.Count > 0))
            {
                quality += 1;
            }

            if (searchSettings.SearchTags != null && searchSettings.SearchTags.Count > 0)
            {
                quality += 1;
            }

            if (avatar != null && (!string.IsNullOrEmpty(avatar.Url) || !string.IsNullOrEmpty(avatar.Filename)))
            {
                quality += 2;
            }

            return quality;
        }

        public static GroupSearchModel FromData(Group group, GroupSearchSettings settings, GroupMember owner, Avatar groupAvatar, Avatar groupCover, ExternalCommunity community = null)
        {
            return new GroupSearchModel
            {
                GroupID = group.GroupID,
                IsDeleted = group.IsDeleted || !settings.IsSearchable,
                MemberCount = group.MemberCount,
                GroupTitle = group.Title,
                IsPublic = group.IsPublic,
                OwnerUserID = group.OwnerID,
                Tags = settings.SearchTags != null ? settings.SearchTags.Select(i => (GroupSearchTag)i).ToArray() : new GroupSearchTag[0],
                Description = settings.Description,
                OwnerUsername = owner == null ? null : owner.GetTitleName(),
                MatchAllGames = settings.MatchAllGames,
                Games = settings.Games != null ? settings.Games.ToArray() : new int[0],
                Subtype = group.Subtype,
                IsStreaming = group.IsStreaming,
                StreamingTimestamp = group.StreamingTimestamp,
                IsFeatured = group.IsFeatured,
                FeaturedTimestamp = group.FeaturedTimestamp,
                Quality = CalculateQuality(group, settings, groupAvatar, community),
                FlaggedAsInappropriate = group.FlaggedAsInappropriate,
                CreationTimestamp = group.DateCreated.SafeToEpochMilliseconds()
            };
        }
    }
}
