﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.Data
{
    public static class TwitchModelHelper
    {
        private static readonly LogCategory Logger = new LogCategory("TwitchModelHelper");
        private static readonly LogCategory DiagLogger = new LogCategory("TwitchModelHelper") { Throttle = TimeSpan.FromSeconds(60) };

        public static ExternalCommunity CreateOrUpdateCommunity(string channelID, ExternalAccount account = null, int regionID = 0)
        {            
            var community = ExternalCommunity.GetLocal(channelID, AccountType.Twitch);
            if (community == null)
            {
                var channel = GetChannelByID(channelID);
                community = CreateCommunity(channel, account, regionID);
            }
            else if (account!=null || community.ShouldSyncInfo())
            {
                var channel = GetChannelByID(channelID);
                UpdateCommunity(community, channel, account);
            }

            return community;
        }

        public static void CreateOrUpdateCommunityRoles(string channelID, string channelName)
        {
            var tags = new[] {GroupRoleTag.SyncedOwner, GroupRoleTag.SyncedModerator, GroupRoleTag.SyncedSubscriberTier3, GroupRoleTag.SyncedSubscriberTier2, GroupRoleTag.SyncedSubscriber, GroupRoleTag.SyncedFollower};
            var roles = ExternalCommunityRole.MultiGetLocal(tags.Select(t => new KeyInfo(channelID, AccountType.Twitch, t))).ToDictionary(r => r.RoleTag);

            foreach (var tag in tags)
            {
                ExternalCommunityRole role;
                if (roles.TryGetValue(tag, out role))
                {
                    continue;
                }

                role = new ExternalCommunityRole
                {
                    ExternalID = channelID,
                    Type = AccountType.Twitch,
                    RoleTag = tag,
                    VanityColor = 0,
                    VanityBadge = string.Empty,
                    IsPremium = IsPremiumRole(tag)
                };

                role.InsertLocal();
            }
        }

        private static bool IsPremiumRole(GroupRoleTag tag)
        {
            return tag == GroupRoleTag.SyncedSubscriber || tag == GroupRoleTag.SyncedSubscriberTier2 || tag == GroupRoleTag.SyncedSubscriberTier3;
        }

        public static ExternalAccount CreateOrUpdateAccount(string code, string state, string clientID, string clientSecret, string redirectUrl)
        {
            var tokenResponse = TwitchApiHelper.GetClient(clientID).GetTwitchToken(code, state, redirectUrl);
            if (tokenResponse.Status != TwitchResponseStatus.Success)
            {
                Logger.Warn("Unable to sync Twitch account. Token Error.", tokenResponse);
                throw new DataValidationException("Twitch token error");
            }
            var tokenInfo = tokenResponse.Value;

            var rootResponse = TwitchApiHelper.Default.GetTwitchRootInfo(tokenInfo.AccessToken);
            if (rootResponse.Status != TwitchResponseStatus.Success)
            {
                Logger.Warn("Unable to sync Twitch account. Root Info Error.", rootResponse);
                throw new DataValidationException("Twitch token error");
            }
            var rootInfo = rootResponse.Value;

            if (!rootInfo.Token.Valid)
            {
                Logger.Warn("Unable to sync Twitch account. The token is invalid.");
                throw new DataValidationException("Twitch token error");
            }

            // Get Account Details from the Twitch API
            var userResponse = TwitchApiHelper.Default.GetUser(tokenInfo.AccessToken);
            if (userResponse.Status!=TwitchResponseStatus.Success)
            {
                Logger.Warn("Failed to retreive Twitch account", userResponse);
                throw new DataValidationException("Twitch user access error");
            }
            var user = userResponse.Value;

            var account = ExternalAccount.GetLocal(user.ID, AccountType.Twitch);
            if (account == null)
            {
                account = new ExternalAccount
                {
                    ExternalID = user.ID,
                    Type = AccountType.Twitch,
                    LastSyncTime = DateTime.MinValue,
                    IsPartnered = user.Partnered,
                    AvatarUrl = user.Logo ?? string.Empty,
                    AuthToken = tokenInfo.AccessToken,
                    RefreshToken = tokenInfo.RefreshToken,
                    Scopes = new HashSet<string>(tokenInfo.Scope),
                    ExternalUsername = user.Name,
                    ExternalDisplayName = user.DisplayName,
                    MappedUsers = new HashSet<int>(),
                    ClientID = clientID
                };

                account.InsertLocal();

                Logger.Trace("Created ExternalAccount", account);
            }
            else
            {
                account.AuthToken = tokenInfo.AccessToken;
                account.RefreshToken = tokenInfo.AccessToken;
                account.Scopes = new HashSet<string>(tokenInfo.Scope);
                account.NeedsReauthentication = false;
                account.ClientID = clientID;
                account.Update(a => a.AuthToken, a => a.RefreshToken, a => a.Scopes, a => a.NeedsReauthentication, a => a.ClientID);
                UpdateAccount(account, user);

                Logger.Trace("Updated ExternalAccount", account);
            }

            return account;
        }

        public static void UpdateAccount(ExternalAccount account, TwitchApi.User passed = null)
        {
            var user = passed;
            if (user == null)
            {
                var response = TwitchApiHelper.Default.GetUser(account.AuthToken);
                if (response.Status!=TwitchResponseStatus.Success)
                {
                    if (response.Status == TwitchResponseStatus.Unauthorized)
                    {
                        MarkForReauthorization(account, "GetUser");
                    }

                    throw new InvalidOperationException("Twitch user access error. " + response.Status);
                }
                user = response.Value;
            }

            var expressions = new List<Expression<Func<ExternalAccount, object>>>();
            if (account.IsPartnered != user.Partnered)
            {
                account.IsPartnered = user.Partnered;
                expressions.Add(a => a.IsPartnered);
            }

            var newAvatar = user.Logo ?? string.Empty;
            if (account.AvatarUrl != newAvatar)
            {
                account.AvatarUrl = newAvatar;
                expressions.Add(a => a.AvatarUrl);
            }

            if (account.ExternalUsername != user.Name)
            {
                account.ExternalUsername = user.Name;
                expressions.Add(a => a.ExternalUsername);
            }

            if (account.ExternalDisplayName != user.DisplayName)
            {
                account.ExternalDisplayName = user.DisplayName;
                expressions.Add(a => a.ExternalDisplayName);
            }

            if (expressions.Any())
            {
                account.Update(expressions.ToArray());
            }
        }

        public static ExternalCommunity GetCommunityFromFollow(Follow follow)
        {
            if (follow == null || follow.Channel == null)
            {
                return null;
            }

            var externalCommunity = ExternalCommunity.GetLocal(follow.Channel.ID, AccountType.Twitch);
            if (externalCommunity != null)
            {
                return externalCommunity;
            }

            var channel = GetChannelByID(follow.Channel.ID, false);
            if (channel == null)
            {
                return null;
            }

            return CreateCommunity(channel);
        }

        public static ExternalCommunity GetOrCreateCommunity(string externalID)
        {
            var externalCommunity = ExternalCommunity.GetLocal(externalID, AccountType.Twitch);            
            if (externalCommunity != null)
            {
                return externalCommunity;
            }

            return CreateCommunity(GetChannelByID(externalID));
        }

        private static ExternalCommunity CreateCommunity(Channel channel, ExternalAccount account = null, int regionID = 0)
        {
            int? subscribers = null;
            var isPartner = channel.Partner || channel.BroadcasterType == "partner";
            var isAffiliate = channel.BroadcasterType == "affiliate";
            if ((isPartner || isAffiliate) && account != null)
            {
                var subscriptionsResponse = TwitchApiHelper.GetClient(account.ClientID).GetSubscriptions(channel.ID, account.AuthToken, 0, 1);
                if (subscriptionsResponse.Status == TwitchResponseStatus.Success)
                {
                    subscribers = subscriptionsResponse.Value.Total;
                }
                else if (subscriptionsResponse.Status == TwitchResponseStatus.Unauthorized)
                {
                    MarkForReauthorization(account, "GetSubscriptions");
                }
            }

            var community = ExternalCommunity.GetLocal(channel.ID, AccountType.Twitch);

            if (community == null)
            {
                community = new ExternalCommunity
                {
                    ExternalID = channel.ID,
                    ExternalName = channel.Name,
                    ExternalDisplayName = channel.DisplayName,
                    MappedGroups = new HashSet<Guid>(),
                    Type = AccountType.Twitch,
                    AvatarUrl = channel.Logo ?? string.Empty,
                    IsPartnered = isPartner,
                    IsAffiliate = isAffiliate,
                    RegionID = regionID,
                    ExternalDateCreated = channel.CreatedAt.ToUniversalTime(),
                    Followers = channel.Followers,
                    Subscribers = subscribers ?? 0,
                    ExternalGameName = channel.Game,
                    GameID = GetCurseGame(channel.Game),
                    ExternalStatus = channel.Status,
                    SyncedTimestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                };

                community.InsertLocal();
                ExternalCommunityIndexWorker.Create(community);
                Logger.Trace("Created ExternalCommunity", community);
            }

            CreateOrUpdateCommunityRoles(community.ExternalID, community.ExternalName);

            return community;
        }

        public static void UpdateCommunity(ExternalCommunity community, ExternalAccount account = null)
        {
            var channel = GetChannelByID(community.ExternalID, false);
            if (channel == null)
            {
                return;
            }

            UpdateCommunity(community, channel, account);
        }

        public static void UpdateCommunity(ExternalCommunity community, Channel channel, ExternalAccount account = null)
        {
            var reindex = false;
            int? subscribers = null;
            var isPartner = channel.Partner || channel.BroadcasterType == "partner";
            var isAffiliate = channel.BroadcasterType == "affiliate";
            if ((isPartner || isAffiliate) && account != null)
            {
                var subscriptionsResponse = TwitchApiHelper.GetClient(account.ClientID).GetSubscriptions(channel.ID, account.AuthToken, 0, 1);
                if (subscriptionsResponse.Status == TwitchResponseStatus.Success)
                {
                    subscribers = subscriptionsResponse.Value.Total;
                }
                else if (subscriptionsResponse.Status == TwitchResponseStatus.Unauthorized)
                {
                    MarkForReauthorization(account, "GetSubscriptions");
                }
            }

            var expressions = new List<Expression<Func<ExternalCommunity, object>>>();

            if (community.IsPartnered != isPartner)
            {
                community.IsPartnered = isPartner;
                expressions.Add(c => c.IsPartnered);
            }

            if(community.IsAffiliate != isAffiliate)
            {
                community.IsAffiliate = isAffiliate;
                expressions.Add(c => c.IsAffiliate);
            }

            if (community.AvatarUrl != channel.Logo && !(community.AvatarUrl == string.Empty && channel.Logo == null))
            {
                var newAvatar = channel.Logo ?? string.Empty;
                community.AvatarUrl = newAvatar;
                expressions.Add(c => c.AvatarUrl);

                reindex = true;
                UpdateAvatar(AvatarType.SyncedCommunity, community.GetAvatarEntityID(), newAvatar);
            }

            if (community.ExternalName != channel.Name && !string.IsNullOrWhiteSpace(channel.Name))
            {
                community.ExternalName = channel.Name;
                expressions.Add(c => c.ExternalName);
                reindex = true;
            }

            if (community.ExternalDisplayName != channel.DisplayName && !string.IsNullOrWhiteSpace(channel.DisplayName))
            {
                community.ExternalDisplayName = channel.DisplayName;
                expressions.Add(c => c.ExternalDisplayName);
                reindex = true;
            }

            if (community.Followers != channel.Followers)
            {
                community.Followers = channel.Followers;
                expressions.Add(c => c.Followers);
            }

            if (subscribers.HasValue && community.Subscribers != subscribers.Value)
            {
                community.Subscribers = subscribers.Value;
                expressions.Add(c => c.Subscribers);
            }

            if (channel.Status != community.ExternalStatus && (!string.IsNullOrEmpty(channel.Status) || string.IsNullOrEmpty(community.ExternalStatus)))
            {
                community.ExternalStatus = channel.Status ?? string.Empty;
                expressions.Add(c => c.ExternalStatus);
                reindex = true;
            }

            if (channel.Game != community.ExternalGameName && (!string.IsNullOrEmpty(channel.Game) || string.IsNullOrEmpty(community.ExternalGameName)))
            {
                community.ExternalGameName = channel.Game ?? string.Empty;
                community.GameID = GetCurseGame(channel.Game);
                expressions.Add(c => c.ExternalGameName);
                expressions.Add(c => c.GameID);
                reindex = true;
            }

            reindex |= community.ShouldSyncInfo();
            community.SyncedTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
            expressions.Add(c => c.SyncedTimestamp);

            community.Update(expressions.ToArray());
            if (reindex)
            {
                ExternalCommunityIndexWorker.Create(community);
            }

            CreateOrUpdateCommunityRoles(community.ExternalID, community.ExternalDisplayName);

            Logger.Trace("Updated ExternalCommunity: " + community.ExternalName, community);
        }

        private static void UpdateAvatar(AvatarType type, string entityID, string newUrl)
        {
            var avatar = Avatar.GetByTypeAndID(type, entityID);
            if (avatar == null && !string.IsNullOrEmpty(newUrl))
            {
                avatar = new Avatar
                {
                    AvatarType = (int)type,
                    EntityID = entityID,
                    Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                    Url = newUrl
                };
                avatar.InsertLocal();
            }
            else if(avatar != null && avatar.Url != newUrl)
            {
                avatar.Url = newUrl;
                avatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
                avatar.Update(a => a.Url, a => a.Timestamp);
            }
        }

        private static int GetCurseGame(string externalGameName)
        {
            // TODO: Figure out curse internal game ID if exists
            return 0;
        }

        private static Channel GetChannelByID(string externalID, bool throwOnNonSuccess = true)
        {
            var channel = TwitchApiHelper.Default.GetChannel(externalID);

            if (channel.Status!=TwitchResponseStatus.Success)
            {
                if (!throwOnNonSuccess)
                {
                    return null;
                }

                if (channel.Status != TwitchResponseStatus.Unprocessable && channel.Status != TwitchResponseStatus.NotFound)
                {
                    Logger.Warn("Failed to retrieve channel!", channel);
                }

                throw new DataNotFoundException(channel.Value != null
                    ? string.IsNullOrEmpty(channel.Value.ErrorMessage) ? "Unspecified error with status code: " + channel.StatusCode : channel.Value.ErrorMessage
                    : "Unexpected status code: " + channel.StatusCode);
            }

            return channel.Value;
        }

        public static bool TestUnauthorizedTokenValidity(string clientID, string token, string source = null)
        {
            var root = TwitchApiHelper.GetClient(clientID).GetTwitchRootInfo(token);
            if (root.Status != TwitchResponseStatus.Success)
            {
                // Assume API error, assume token is fine
                return true;
            }

            if (!root.Value.Token.Valid)
            {
                // Truly invalid
                return false;
            }

            if (source != null)
            {
                DiagLogger.Warn($"Token was valid but unauthorized to access {source}", new { response = root.Value });
            }
            return true;
        }

        public static void MarkForReauthorization(ExternalAccount account, string source = null)
        {
            if(!TestUnauthorizedTokenValidity(account.ClientID, account.AuthToken, source))
            {
                account.MarkForReauthentication();

                if (account.MergedUserID > 0)
                {
                    var userStats = UserStatistics.GetByUserID(account.MergedUserID);
                    if (userStats == null)
                    {
                        return;
                    }

                    var timestamp = DateTime.UtcNow;
                    var timestampMs = timestamp.ToEpochMilliseconds();
                    // Only move the timestamp forward
                    if (userStats.TokenTimestamp < timestampMs)
                    {
                        userStats.TokenTimestamp = timestampMs;
                        userStats.Update(p => p.TokenTimestamp);
                    }

                }
            }
        }
    }
}
