﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using System.Threading;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.DataMigrations
{
    class Program
    {
        static void Main(string[] args)
        {
            Console.Write("Initializing configuration...");
            StorageConfiguration.Initialize("CurseVoice", "US-East", ConfigurationMode.Release, ConfigurationServices.Database | ConfigurationServices.Queue);
            Console.WriteLine("Done!");

            


            try
            {
                CheckGroupMemberIndexIntegrity(new Guid(" b169ca29-e483-4298-b26e-7f2d51d861d0"));
                //RenameUser();

                //new Thread(FixVoiceInviteCodes) { IsBackground = true, Priority = ThreadPriority.Highest }.Start();

                //Console.WriteLine("Which region code to import?");
                //var region = Console.ReadLine();
                //new Thread(CleanGroupMembers) {IsBackground = true}.Start();
                //new Thread(() => MigrateGroups(region)) { IsBackground = true, Priority = ThreadPriority.Highest}.Start();
                //new Thread(() => RemoveBins("cursevoice-global", "Group", new HashSet<string> {"Members"})) { IsBackground = true }.Start();
                //new Thread(() => DebugBinNames("cursevoice-global", "Group")) { IsBackground = true }.Start();

                Console.WriteLine("Press the ESC key to stop the import...");
                while (true)
                {
                    if (Console.ReadKey().Key == ConsoleKey.Escape)
                    {
                        _keepProcessing = false;
                        break;
                    }

                }
            }
            catch (Exception)
            {

            }
            finally
            {
                StorageConfiguration.Shutdown();
                Console.WriteLine("Done!");
                Console.ReadLine();
            }
        }

       

        private static volatile bool _keepProcessing = true;

        private static volatile int _membersCreated = 0;
        private static volatile int _membersUpdated = 0;
        private static volatile int _membersSkipped = 0;

        static void RenameUser()
        {
            while (true)
            {
                Console.WriteLine("UserID: ");
                var userID = int.Parse(Console.ReadLine());
                Console.WriteLine("Username: ");
                var username = Console.ReadLine();

                var userRegion = UserRegion.GetByUserID(userID, true);
                var user = userRegion.GetUser();
                user.Username = username;
                user.Update(p => p.Username);

                var memberships = GroupMember.GetAllByUserID(userID);

                //Groups
                foreach (var member in memberships.Where(p => !string.IsNullOrWhiteSpace(p.Username)))
                {
                    member.Username = username;
                    member.Update(p => p.Username);

                    var group = Group.GetByID(member.GroupID);
                    GroupMemberIndexWorker.CreateFullSync(group);
                    GroupChangeCordinator.UpdateUsers(group, member.UserID, new HashSet<int> {member.UserID});
                }

                // Friends
                foreach (var region in Friendship.AllConfigurations)
                {
                    var friends = Friendship.GetAll(region, p => p.OtherUserID, userID);
                    foreach (var friend in friends)
                    {
                        friend.OtherUserNickname = username;
                        friend.OtherUsername = username;
                        friend.DateMessaged = DateTime.UtcNow;
                        friend.Update(p => p.OtherUserNickname, p => p.OtherUsername, p => p.DateMessaged);
                    }
                }

                RegionalUserChangeResolver.Create(userID);



                var existing = FriendHint.GetAllLocal(p => p.UserID, userID);

                foreach (var hint in existing)
                {
                    if (hint.Type == FriendHintType.Username)
                    {
                        hint.Delete();
                    }
                }

                var friendHint = new FriendHint
                {
                    UserID = userID,
                    Type = FriendHintType.Username,
                    SearchTerm = username,
                    AvatarUrl = user.AvatarUrl,
                    Verification = FriendHintVerification.Verified
                };
                friendHint.InsertLocal();

                // Queue up a job to reindex the user
                FriendHintSearchIndexer.CreateForUser(userID);

                Console.WriteLine("Done!");
                Console.ReadLine();
                Console.Clear();
            }

        }
       

        static void RemoveBins(string ns, string setName, HashSet<string> binNames)
        {
            var counter = 0;
            var config = GroupMember.LocalConfiguration;
            var binList = binNames.Select(Bin.AsNull).ToArray();

            config.Client.ScanAll(GetDefaultScanPolicy(false), ns, setName, (k, r) =>
            {
                if (!_keepProcessing)
                {
                    throw new AerospikeException.ScanTerminated();
                }
                if (++counter % 10 == 0)
                {
                    Console.Title = counter.ToString("N0");
                }


                config.Client.Put(null, k, binList);
            });

        }

        static void DebugBinNames(string ns, string setName)
        {
            var counter = 0;
            var config = GroupMember.LocalConfiguration;
            var distinctBinNames = new HashSet<string>();
            config.Client.ScanAll(GetDefaultScanPolicy(true), ns, setName, (k, r) =>
            {
                if (!_keepProcessing)
                {
                    throw new AerospikeException.ScanTerminated();
                }
                if (++counter % 10 == 0)
                {
                    Console.Title = counter.ToString("N0");
                }

                foreach (var bin in r.bins)
                {
                    if (!distinctBinNames.Contains(bin.Key))
                    {
                        distinctBinNames.Add(bin.Key);
                        Console.WriteLine("New bin name: " + bin.Key);
                    }
                }
            });

        }

        static void MigrateGroups(string sourceRegion)
        {
            AerospikeConfiguration membersConfiguration;

            switch (sourceRegion)
            {
                case "na":
                    membersConfiguration = new AerospikeConfiguration
                    {
                        Addresses = new[] { "nosql01a-iad.curse.io", "nosql01b-iad.curse.io", "nosql01c-iad.curse.io" },
                        Port = 3000,
                        Password = "",
                        RegionIdentifier = 1,
                        RegionGroup = "NA",
                        RegionKey = "US-East",
                        Username = ""
                    };
                    break;

                case "eu":
                    membersConfiguration = new AerospikeConfiguration
                    {
                        Addresses = new[] { "nosql01a-dub.curse.io", "nosql01b-dub.curse.io", "nosql01c-dub.curse.io" },
                        Port = 3000,
                        Password = "",
                        RegionIdentifier = 3,
                        RegionGroup = "EU",
                        RegionKey = "EU-West",
                        Username = ""
                    };
                    break;

                case "ap":
                    membersConfiguration = new AerospikeConfiguration
                    {
                        Addresses = new[] { "nosql01a-sin.curse.io", "nosql01b-sin.curse.io" },
                        Port = 3000,
                        Password = "",
                        RegionIdentifier = 5,
                        RegionGroup = "AP",
                        RegionKey = "AP-Southeast",
                        Username = ""
                    };
                    break;
                default:
                    Console.WriteLine("Unknown region: " + sourceRegion);
                    return;
            }
           
            Console.Write("Connecting to region: " + sourceRegion);
            membersConfiguration.Connect();
            Console.WriteLine("Done!");

            var sourceConfiguration = new AerospikeConfiguration
            {
                Addresses = new[] { "nosql02a-iad.curse.io", "nosql02b-iad.curse.io", "nosql02c-iad.curse.io" },
                Port = 3000,
                Password = "",
                RegionIdentifier = 1,
                RegionGroup = "NA",
                RegionKey = "US-East",
                Username = ""
            };

            sourceConfiguration.Connect();

            var destinationConfiguration = new AerospikeConfiguration
            {
                Addresses = new[] { "nosql03a-iad.curse.io", "nosql03b-iad.curse.io", "nosql03c-iad.curse.io" },
                Port = 3000,
                Password = "",
                RegionIdentifier = 1,
                RegionGroup = "NA",
                RegionKey = "US-East",
                Username = ""
            };

            destinationConfiguration.Connect();

        }

        static void FixVoiceInviteCodes()
        {            

            var destinationConfiguration = new AerospikeConfiguration
            {
                Addresses = new[] { "nosql03a-iad.curse.io", "nosql03b-iad.curse.io", "nosql03c-iad.curse.io" },
                Port = 3000,
                Password = "",
                RegionIdentifier = 1,
                RegionGroup = "NA",
                RegionKey = "US-East",
                Username = ""
            };

            destinationConfiguration.Connect();

            DoFixVoiceInviteCodes(destinationConfiguration);
        }

       


        static void DoFixVoiceInviteCodes(AerospikeConfiguration destinationConfiguration)
        {

            try
            {
                var groupKeys = new ConcurrentQueue<Key>();
                var destinationKeys = new HashSet<Key>();
                Console.Write("Scanning source group keys...");


                destinationConfiguration.Client.ScanAll(GetDefaultScanPolicy(false), "cursevoice-global", "Group", (k, r) =>
                {
                    if (!_keepProcessing)
                    {
                        throw new AerospikeException.ScanTerminated();
                    }

                    groupKeys.Enqueue(k);

                });

                Console.WriteLine("Done! " + groupKeys.Count.ToString("N0") + " keys read!");
                
                Key key;
                var total = groupKeys.Count;
                var counter = 0;
                var skippedCounter = 0;
                var updatedCounter = 0;
                while (groupKeys.TryDequeue(out key))
                {
                    if (!_keepProcessing)
                    {
                        return;
                    }

                    if (++counter % 100 == 0)
                    {
                        Console.Title = "Processing group " + counter.ToString("N0") + " of " + total.ToString("N0") + " (Updated: " + updatedCounter.ToString("N0") + ", Skipped: " + skippedCounter.ToString("N0") + ")";
                    }

                    try
                    {

                        var group = Group.GetByKey(destinationConfiguration, key);
                        if (group.VoiceSessionCode != null && group.VoiceSessionCode.Length > 10)
                        {
                            group.VoiceSessionCode = string.Empty;
                            group.Update(p => p.VoiceSessionCode);
                        }
                        else
                        {
                            ++skippedCounter;
                        }

                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine("Failed: " + ex.Message);
                        Console.WriteLine("Stack: " + ex.StackTrace);
                        Console.WriteLine();
                    }
                }

            }
            catch (AerospikeException.ScanTerminated)
            {
                Console.WriteLine("Process stopped!");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed: " + ex.Message);
            }
        }


        static ScanPolicy GetDefaultScanPolicy(bool includeBinData)
        {
            return new ScanPolicy
            {                
                concurrentNodes = true,
                consistencyLevel = ConsistencyLevel.CONSISTENCY_ONE,
                failOnClusterChange = false,
                includeBinData = includeBinData,
                maxConcurrentNodes = 0,
                timeout = 10000,
                maxRetries = 1,
                priority = Priority.HIGH
            };
        }

        static void ValidateIndexes<T>() where T : BaseTable<T>, new()
        {
            try
            {
                BaseTable<T>.GetLocal(1);
            }
            catch (Exception ex)
            {

            }

        }

        static void CheckGroupMemberIndexIntegrity(Guid groupID)
        {
            
            // Perform a full index query, and look for a known value
            var group = Group.GetByID(groupID);

            if (group == null)
            {
                Console.WriteLine("Failed to retrieve group from database");
                Console.ReadKey(true);
                return;
            }

            var indexMembers = group.GetAllMembers().ToDictionary(p => p.UserID);
            var readMembers = new List<GroupMember>(indexMembers.Count);
            var userID = 0;
            var maxUserID = 26469169;

            while (true)
            {
                var currentSet = new HashSet<int>();
                for (var i = 0; i < 10000; i++)
                {
                    currentSet.Add(userID++);                    
                }

                var setMembers = GroupMember.MultiGetLocal(currentSet.Select(p => new KeyInfo(groupID, p)));
                readMembers.AddRange(setMembers);

                if (userID >= maxUserID)
                {
                    break;
                }
            }

            foreach (var member in readMembers)
            {
                if (!indexMembers.ContainsKey(member.UserID))
                {
                    Console.WriteLine("Member missing from index! " + member.Username + " (" + member.UserID + ")");
                }
            }

            Console.ReadLine();
        }
    }
}
