﻿using System;
using System.IO;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.FilesWebService.Manager;
using Curse.Friends.FilesWebService.Models;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.FilesWebService.Controllers
{
    /// <summary>
    /// Handles actions related to files.
    /// </summary>
    [ExcludeFromGenerator]
    public class FileController : MicroServiceUploadController
    {
        /// <summary>
        /// Gets the specified file.
        /// </summary>
        /// <param name="fileID">The ID of the file.</param>
        /// <param name="filename">The name of the file.</param>
        /// <returns>The file.</returns>
        [HttpGet]
        [Route("{fileID}/{filename}")]
        [ResponseType(typeof(Stream))]
        [AuthenticationFilter(AuthenticationLevel.LoggedIn, true)]
        public IHttpActionResult Get(Guid fileID, string filename)
        {
            var file = FileManager.GetFile(Token.UserID, fileID, filename);
            if (file == null || file.Metadata.Filename != filename)
            {
                return NotFound();
            }

            var response = new HttpResponseMessage
            {
                StatusCode = HttpStatusCode.OK,
                Content = new StreamContent(file.Content),
            };
            response.Content.Headers.ContentType = new MediaTypeHeaderValue(file.Metadata.ContentType);
            response.Content.Headers.ContentDisposition = new ContentDispositionHeaderValue("attachment")
            {
                FileName = file.Metadata.Filename
            };
            return ResponseMessage(response);
        }

        /// <summary>
        /// Gets the metadata for the specified file.
        /// </summary>
        /// <param name="fileID">The ID of the file.</param>
        /// <param name="filename">The name of the file.</param>
        /// <returns>The file's metadata.</returns>
        [HttpGet]
        [Route("{fileID}/{filename}/metadata")]
        [ResponseType(typeof(FileMetadata))]
        [AuthenticationFilter(AuthenticationLevel.LoggedIn, true)]
        public IHttpActionResult GetMetadata(Guid fileID, string filename)
        {
            return Ok(FileManager.GetFileMetadata(Token.UserID, fileID, filename));
        }

        /// <summary>
        /// Deletes the specified file.
        /// </summary>
        /// <param name="fileID">The ID of the file.</param>
        /// <param name="filename">The name of the file.</param>
        /// <returns>Status code based on success or failure.</returns>
        [HttpDelete]
        [Route("{fileID}/{filename}")]
        [ResponseType(typeof(void))]
        [SocialBanFilter]
        public IHttpActionResult Delete(Guid fileID, string filename)
        {
            if (!FileManager.DeleteFile(Token.UserID, fileID, filename))
            {
                return NotFound();
            }

            return StatusCode(HttpStatusCode.NoContent);
        }

        /// <summary>
        /// Uploads a file to the conversation between the calling user and the user with the specified friend ID.
        /// </summary>
        /// <param name="friendID">The ID of the friend.</param>
        /// <returns>The info for the uploaded file.</returns>
        [HttpPost]
        [Route("users/{friendID}")]
        [ResponseType(typeof(AttachmentNotification))]
        [ExcludeFromGenerator]
        [SocialBanFilter]
        public AttachmentNotification ShareWithFriend(int friendID)
        {            
            var file = GetPostedFile();
            return FileManager.UploadFile(Token.UserID, friendID, SharedFile.FromUploadedFile(Token.UserID, file)).ToNotification();
        }

        /// <summary>
        /// Uploads a file to the group conversation.
        /// </summary>
        /// <param name="groupID">The group to upload the file to.</param>
        /// <returns>The info for the uploaded file.</returns>
        [HttpPost]
        [Route("groups/{groupID}")]
        [ResponseType(typeof(AttachmentNotification))]
        [ExcludeFromGenerator]
        [SocialBanFilter]
        public AttachmentNotification ShareWithGroup(Guid groupID)
        {
            var file = GetPostedFile();            
            return FileManager.UploadFile(Token.UserID, groupID, SharedFile.FromUploadedFile(Token.UserID, file)).ToNotification();
        }        
    }
}