﻿using System;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.FilesWebService.Configuration;
using Curse.Friends.FilesWebService.Models;
using Curse.Friends.MicroService.Exceptions;
using Curse.Logging;

namespace Curse.Friends.FilesWebService.Manager
{
    public class FileManager
    {
        private static readonly TimeSpan ThrottleWindow = TimeSpan.FromMinutes(FileServiceConfiguration.Global.ThrottleWindowMinutes);
        private static readonly int MaxRequestsPerWindow = FileServiceConfiguration.Global.MaxRequestsPerWindow;
        private static readonly long MaxSizePerWindow = FileServiceConfiguration.Global.MaxSizePerWindow;

        public static Attachment UploadFile(int requestorID, int friendID, SharedFile file)
        {
            VerifyThrottleLimits(requestorID, file);

            var requestor = GetUserRegion(requestorID);
            if (requestor == null)
            {
                throw new FriendshipPermissionException();
            }

            var friendship = Friendship.Get(requestor.RegionID, requestor.UserID, friendID);
            if (friendship == null || friendship.Status != FriendshipStatus.Confirmed)
            {
                throw new FriendshipPermissionException();
            }

            return DoUploadFile(requestor, PrivateConversation.GenerateConversationID(requestorID, friendID), file);
        }

        public static Attachment UploadFile(int requestorID, Guid groupID, SharedFile file)
        {
            VerifyThrottleLimits(requestorID, file);

            var requestor = GetUserRegion(requestorID);
            if (requestor == null)
            {
                throw new GroupPermissionException(GroupPermissions.ChatAttachFiles);
            }

            var group = Group.GetLocal(groupID);
            if (group == null)
            {
                throw new GroupPermissionException(GroupPermissions.ChatAttachFiles);
            }

            group.CheckPermission(GroupPermissions.ChatAttachFiles, requestorID);

            return DoUploadFile(requestor, groupID.ToString(), file);
        }

        private static Attachment DoUploadFile(UserRegion requestor, string conversationID, SharedFile file)
        {
            VirusScanner.Scan(file.Content);

            var bucket = FileServiceConfiguration.Global.Buckets.FirstOrDefault(r => r.RegionIdentifier == requestor.RegionID);
            if (bucket == null)
            {
                bucket = FileServiceConfiguration.Current;
                Logger.Warn("No bucket was found for the user's region, defaulting to the current region's bucket.", new { userRegion = requestor, conversationID, CurrentRegion = bucket.RegionIdentifier });
            }

            // Upload to S3
            var key = S3Manager.SaveFile(bucket, file.Metadata, file.Content);

            // Put it in Aerospike
            var attachment = new Attachment
            {
                FileID = key,
                ConversationID = conversationID,
                Status = AttachmentStatus.Available,
                Url = string.Format(bucket.UrlFormat, key, file.Metadata.Filename),
                StorageRegionID = bucket.RegionIdentifier,

                // Metadata
                DateUploaded = file.Metadata.DateUploaded,
                Filename = file.Metadata.Filename,
                FileSize = file.Metadata.FileSize,
                UploaderUserID = file.Metadata.UploaderUserID, 
                FileType = file.Metadata.ContentType, 
            };
            attachment.InsertLocal();

            return attachment;
        }

        public static SharedFile GetFile(int requestorID, Guid fileID, string filename)
        {
            Attachment attachment;
            if (!VerifyAttachmentAccess(requestorID, fileID, filename, out attachment))
            {
                return null;
            }

            FileMetadata metadata;
            var stream = S3Manager.GetFile(FileServiceConfiguration.Current, fileID.ToString(), out metadata);
            if (stream == null || metadata.Filename != filename)
            {
                Logger.Debug("Incorrect filename requested for the shared file", new {fileID, filename});
                return null;
            }
            return new SharedFile(metadata, stream);
        }

        public static FileMetadata GetFileMetadata(int requestorID, Guid fileID, string filename)
        {
            Attachment attachment;
            if (!VerifyAttachmentAccess(requestorID, fileID, filename, out attachment))
            {
                return null;
            }

            return S3Manager.GetFileMetadata(FileServiceConfiguration.Current, fileID.ToString());
        }

        public static bool DeleteFile(int requestorID, Guid fileID, string filename)
        {
            Attachment attachment;
            if (!VerifyAttachmentAccess(requestorID, fileID, filename, out attachment))
            {
                return false;
            }

            try
            {
                S3Manager.DeleteFile(FileServiceConfiguration.Current, fileID.ToString());
                attachment.Delete();
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error deleting shared file.", new {fileID, filename});
                return false;
            }
        }

        private static bool VerifyAttachmentAccess(int requestorID, Guid fileID, string filename, out Attachment attachment)
        {
            var requestor = GetUserRegion(requestorID);
            if (requestor == null)
            {
                attachment = null;
                return false;
            }

            attachment = Attachment.GetLocal(fileID);
            if (attachment == null || attachment.Filename != filename)
            {
                return false;
            }

            if (attachment.UploaderUserID == requestor.UserID)
            {
                return true;
            }

            Guid groupID;
            if (Guid.TryParse(attachment.ConversationID, out groupID))
            {
                var group = Group.GetLocal(groupID);

                if (group == null)
                {
                    throw new GroupPermissionException(GroupPermissions.Access);
                }

                group.CheckPermission(GroupPermissions.Access, requestor.UserID);
            }
            else
            {
                var ids = attachment.ConversationID.Split(':').Select(int.Parse).ToArray();
                if (!ids.Any(u => u == requestor.UserID))
                {
                    throw new FriendshipPermissionException();
                }

                var otherUser = ids.First(u => u != requestor.UserID);
                var friendship = Friendship.Get(requestor.RegionID, requestor.UserID, otherUser);
                if (friendship == null || friendship.Status != FriendshipStatus.Confirmed)
                {
                    throw new FriendshipPermissionException();
                }
            }

            return true;
        }

        private static UserRegion GetUserRegion(int userID)
        {
            var userRegion = UserRegion.GetLocal(userID);
            if (userRegion == null)
            {
                Logger.Debug("Authenticated user does not have a region!", new { userID });
            }
            return userRegion;
        }

        private static void VerifyThrottleLimits(int requestorID, SharedFile file)
        {
            // Ensure file is not too large
            if (file.Metadata.FileSize > Attachment.MaxFileSize)
            {
                throw new FileUploadException(file.Metadata.FileSize, Attachment.MaxFileSize);
            }

            var recentUploads =
                Attachment.GetAllLocal(a => a.UploaderUserID, requestorID)
                    .Where(a => a.DateUploaded > DateTime.UtcNow - ThrottleWindow)
                    .OrderBy(a => a.DateUploaded)
                    .ToArray();

            if (recentUploads.Length > MaxRequestsPerWindow - 1)
            {
                // This request puts you over the rate limit
                var retryDelta = (recentUploads.First().DateUploaded + ThrottleWindow - DateTime.UtcNow);
                throw new FileUploadException(FileUploadFailureReason.TooManyUploads, retryDelta);
            }

            if (recentUploads.Aggregate((long)file.Metadata.FileSize, (current, attachment) => current + attachment.FileSize) > MaxSizePerWindow)
            {
                // This request puts you over the data limit
                var size = file.Metadata.FileSize;
                var oldestPossible = recentUploads.Reverse().TakeWhile(attachment => (size = size + attachment.FileSize) <= MaxSizePerWindow).LastOrDefault();
                throw new FileUploadException(FileUploadFailureReason.TooManyBytes, oldestPossible == null ? null : (TimeSpan?)(oldestPossible.DateUploaded + ThrottleWindow - DateTime.UtcNow));
            }
        }
    }
}