﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using Amazon;
using Amazon.S3;
using Amazon.S3.Model;
using Amazon.S3.Transfer;
using Curse.Friends.FilesWebService.Configuration;

namespace Curse.Friends.FilesWebService.Manager
{
    public static class S3Manager
    {
        public static Guid SaveFile(FileServiceBucketConfiguration bucket, FileMetadata metadata, Stream content)
        {
            var key = Guid.NewGuid();

            using (var client = GetClient(bucket))
            {
                using (var transfer = new TransferUtility(client))
                {
                    var request = new TransferUtilityUploadRequest
                    {
                        BucketName = bucket.BucketName,
                        Key = key.ToString(),
                        InputStream = content,
                        AutoCloseStream = true,
                        ContentType = metadata.ContentType,
                    };

                    foreach (var kvp in metadata.AdditionalMetadata)
                    {
                        request.Metadata[kvp.Key] = kvp.Value;
                    }

                    // Always ensure basic metadata is present regardless of additional metadata values
                    // These will always overwrite any additional metadata values with the same name.
                    request.Metadata[FileMetadata.FilenameKey] = metadata.Filename;
                    request.Metadata[FileMetadata.DateUploadedKey] = metadata.DateUploaded.ToString("O");
                    request.Metadata[FileMetadata.FileSizeKey] = metadata.FileSize.ToString();
                    request.Metadata[FileMetadata.UploaderUserIDKey] = metadata.UploaderUserID.ToString();

                    transfer.Upload(request);
                }
            }

            return key;
        }

        public static void DeleteFile(FileServiceBucketConfiguration bucket, string key)
        {
            using (var client = GetClient(bucket))
            {
                var request = new DeleteObjectRequest
                {
                    BucketName = bucket.BucketName,
                    Key = key
                };

                var response = client.DeleteObject(request);

                if (response.HttpStatusCode != HttpStatusCode.OK)
                {
                    // log
                }
            }
        }

        public static Stream GetFile(FileServiceBucketConfiguration bucket, string key, out FileMetadata metadata)
        {
            metadata = null;
            using (var client = GetClient(bucket))
            {
                var request = new GetObjectRequest
                {
                    BucketName = bucket.BucketName,
                    Key = key
                };

                var response = client.GetObject(request);

                if (response.HttpStatusCode != HttpStatusCode.OK)
                {
                    return null;
                }

                metadata = GetFileMetadataFromHeaders(response.Headers,response.Metadata);

                return response.ResponseStream;
            }
        }

        public static FileMetadata GetFileMetadata(FileServiceBucketConfiguration bucket, string key)
        {
            using (var client = GetClient(bucket))
            {
                var request = new GetObjectMetadataRequest
                {
                    BucketName = bucket.BucketName,
                    Key = key
                };

                var response = client.GetObjectMetadata(request);

                return GetFileMetadataFromHeaders(response.Headers, response.Metadata);
            }
        }

        private static FileMetadata GetFileMetadataFromHeaders(HeadersCollection headers, MetadataCollection metadata)
        {
            var mdDictionary = metadata.ToDictionary();
            mdDictionary.Remove(FileMetadata.UploaderUserIDKey);
            mdDictionary.Remove(FileMetadata.FilenameKey);
            mdDictionary.Remove(FileMetadata.FileSizeKey);
            mdDictionary.Remove(FileMetadata.DateUploadedKey);

            return new FileMetadata(int.Parse(metadata[FileMetadata.UploaderUserIDKey]), metadata[FileMetadata.FilenameKey],
                int.Parse(metadata[FileMetadata.FileSizeKey]), DateTime.Parse(metadata[FileMetadata.DateUploadedKey]),
                headers["Content-Type"], mdDictionary);
        }

        private static AmazonS3Client GetClient(FileServiceBucketConfiguration bucket)
        {
            var region = RegionEndpoint.GetBySystemName(bucket.RegionEndpoint);

            if (region == null)
            {
                throw new Exception("Failed to initialize. No region endpoint found:" + bucket.RegionEndpoint);
            }


            return new AmazonS3Client(bucket.AccessKeyID, bucket.SecretAccessKey, region);
        }

        private static Dictionary<string, string> ToDictionary(this MetadataCollection metadata)
        {
            var dictionary = new Dictionary<string, string>();
            foreach (var key in metadata.Keys)
            {
                dictionary[key.Replace("x-amz-meta-", string.Empty)] = metadata[key];
            }
            return dictionary;
        } 
    }
}